from luigi import Task, Target
from luigi import Parameter as _Parameter
from luigi import IntParameter as _IntParameter
from luigi import DateParameter as _DateParameter
from luigi import FloatParameter as _FloatParameter
from luigi import WrapperTask as _WrapperTask

from functools import partial

from yt.wrapper import YtClient, YPath

from crypta.lib.python.yql.client import create_yql_client
from common.yt_methods import create_table_with_schema
import library.python.resource as lpr

import config
from common import cached_property

import logging

logger = logging.getLogger(__name__)


class TrueTarget(Target):
    def exists(self):
        return True


class Targets(object):
    class YtTarget(Target):
        def __init__(self, yt):
            self.yt = yt

    class PathYtTarget(YtTarget):
        def __init__(self, yt, path):
            Targets.YtTarget.__init__(self, yt)
            self.path = path

    class Exists(PathYtTarget):
        def exists(self):
            return self.yt.exists(self.path)

    class NotExists(PathYtTarget):
        def exists(self):
            return not self.yt.exists(self.path)

    class NotEmpty(PathYtTarget):
        def exists(self):
            _exists = self.yt.exists(self.path)
            if _exists:
                row_count = self.yt.get_attribute(self.path, "row_count", 0)
                _not_empty = row_count > 0
                return _exists and _not_empty
            return _exists

    class TableIsActual(PathYtTarget):
        def __init__(self, yt, path, date, attribute):
            Targets.PathYtTarget.__init__(self, yt, path)
            self.date = date
            self.attribute = attribute

        def exists(self):
            if self.yt.exists(self.path):
                table_attr = self.yt.get(self.path + "/@")
                date_str = table_attr.get(self.attribute, "0001-01-01")
                return self.date <= date_str
            return False

    def __init__(self, yt):
        self.yt = yt

    def exists(self, table):
        return Targets.Exists(self.yt, table)

    def not_exists(self, table):
        return Targets.NotExists(self.yt, table)

    def not_empty(self, table):
        return Targets.NotEmpty(self.yt, table)

    def table_is_actual(self, table, date, attribute="generate_date"):
        return Targets.TableIsActual(self.yt, table, date, attribute)


class BaseTask(Task):
    @cached_property
    def yt(self):
        client = YtClient(proxy=config.Yt.PROXY, token=config.Yt.TOKEN)
        client.targets = Targets(client)
        client.path = YPath
        client.create_table_with_schema = partial(create_table_with_schema, yt=client)
        client.proxy = client.config["proxy"]["url"]
        client.transaction_id = None
        client.token = config.Yt.TOKEN
        client.config["pool"] = config.Yt.POOL
        client.pool = config.Yt.POOL
        return client

    @cached_property
    def yql(self):
        transaction = self.yt.transaction_id
        client = create_yql_client(
            yt_proxy=config.Yt.PROXY,
            token=config.Yql.TOKEN,
            transaction=transaction,
            db=config.Yql.DB,
            yql_server=config.Yql.SERVER,
            yql_port=config.Yql.PORT,
            pool=config.Yt.POOL,
        )
        return client

    @staticmethod
    def render_resource(name, **kwargs):
        return lpr.find(name).format(**kwargs)

    def run(self):
        if hasattr(self, "_run"):
            logger.info("Starting task %s", self.__class__.__name__)
            with self.yt.Transaction() as transaction:
                self.yt.transaction_id = str(transaction.transaction_id)
                logger.info("Running in transaction %s", self.yt.transaction_id)
                self._run()
                self.yt.transaction_id = None
            logger.info("Finishing task %s", self.__class__.__name__)


Parameter = _Parameter
DateParameter = _DateParameter
WrapperTask = _WrapperTask
IntParameter = _IntParameter
FloatParameter = _FloatParameter
