from crypta.graph.fuzzy.lib.luiger import BaseTask, DateParameter, FloatParameter
from crypta.graph.fuzzy.lib.common import cached_property
import crypta.graph.fuzzy.lib.config as conf
from crypta.graph.fuzzy.lib.tasks.classifier.classifier import BayesClassifyTask
import logging

logger = logging.getLogger(__name__)


class ExportToSUP(BaseTask):

    date = DateParameter()
    threshold = FloatParameter()

    @cached_property
    def __classifier(self):
        return BayesClassifyTask(date=self.date)

    def requires(self):
        yield self.__classifier

    @property
    def source(self):
        return self.__classifier.destination

    @property
    def destination(self):
        return conf.Paths.EXPORT_SUP

    @cached_property
    def query(self):
        return self.render_resource(
            name="yql/export_sup",
            source=self.source,
            destination=self.destination,
            date=self.date.isoformat(),
            threshold=self.threshold,
        )

    def output(self):
        yield self.yt.targets.table_is_actual(self.destination, self.date.isoformat())

    def _run(self):
        self.yql.execute(self.query)
        self.yt.set(self.destination + "/@generate_date", self.date.isoformat())


def get_threshold(yt_client):
    for row in yt_client.read_table(conf.Paths.DEBUG_SCORES):
        if (
            row["geo"] == 1
            and row["ssid"] == 1
            and row["login"] == 0
            and row["reqans"] == 0
            and row["households"] == 0
            and row["emails"] == 0
        ):
            threshold = float(row["score"])
            logger.info("Threshold found: %f", threshold)
            return threshold


class ExportExternalSystems(BaseTask):

    date = DateParameter()
    threshold = FloatParameter()

    @cached_property
    def __classifier(self):
        return BayesClassifyTask(date=self.date)

    def requires(self):
        yield self.__classifier

    @property
    def source(self):
        return self.__classifier.destination

    @property
    def destination(self):
        return conf.Paths.EXPORT_EXTERNAL

    @property
    def destination_schema(self):
        return conf.Paths.EXPORT_EXTERNAL_SCHEMA

    def query(self, destination):
        return self.render_resource(
            name="yql/export_external", source=self.source, destination=destination, threshold=self.threshold
        )

    def output(self):
        yield self.yt.targets.table_is_actual(self.destination, self.date.isoformat())

    @staticmethod
    def copy(row):
        output = dict()
        output.update(row)
        yield output

    def _run(self):
        if self.yt.exists(self.destination):
            self.yt.remove(self.destination)
            self.yt.create_table(self.destination)
        with self.yt.TempTable() as tmp:
            self.yql.execute(self.query(tmp))
            self.yt.run_map(self.copy, tmp, self.destination)
        self.yt.set(self.destination + "/@generate_date", self.date.isoformat())


class ExportAll(BaseTask):

    date = DateParameter()

    @cached_property
    def threshold(self):
        return get_threshold(self.yt)

    @cached_property
    def _classifier(self):
        return BayesClassifyTask(date=self.date)

    def requires(self):
        yield self._classifier
        if all(target.exists() for target in self._classifier.output()):
            yield ExportExternalSystems(date=self.date, threshold=self.threshold)
            yield ExportToSUP(date=self.date, threshold=self.threshold)

    def _run(self):
        pass

    def output(self):
        for task in self.requires():
            for target in task.output():
                yield target
