#pragma once

#include <crypta/lib/python/native_yt/cpp/registrar.h>
#include <crypta/lib/python/native_yt/cpp/proto.h>
#include <crypta/graph/fuzzy/lib/tasks/sources/proto/geo.pb.h>
#include <utility>
#include <util/generic/utility.h>
#include <util/generic/vector.h>
#include <util/ysaveload.h>
#include <util/stream/str.h>
#include <util/string/cast.h>
#include <util/generic/maybe.h>
#include <util/generic/xrange.h>
#include "geo_utils.h"

using NYT::IMapper;
using NYT::TNode;
using NYT::IReducer;
using NYT::TTableReader;
using NYT::TTableWriter;
using NNativeYT::TProtoState;

namespace NFuzzyGeo {
    class TSpaceSlicer: public IMapper<TTableReader<TNode>, TTableWriter<TGeoSquare>> {
    public:
        TSpaceSlicer()
            : State()
        {
        }

        TSpaceSlicer(const TBuffer& buffer)
            : State(buffer)
        {
        }

        TMaybe<ui64> GetYandexuid(const TNode& row) const {
            try {
                return FromString<ui64>(row["yandexuid"].AsString());
            } catch(TFromStringException exc) {
                return Nothing();
            }
        }

        bool IsRowValid(const TNode& row) {
            if (not row.HasKey("yandexuid") || not row.HasKey("predicted_home")) {
                return false;
            }
            const auto& homeCoordinates = row["predicted_home"];
            if (not homeCoordinates.HasKey("latitude") || not homeCoordinates.HasKey("longitude")) {
                return false;
            }
            const auto yandexuid = GetYandexuid(row);
            if (not yandexuid.Defined()) {
                return false;
            }
            return true;
        }

        void Do(TTableReader<TNode>* input, TTableWriter<TGeoSquare>* output)
        override {
            for (; input->IsValid(); input->Next()) {
                const auto& row = input->GetRow();
                if (not IsRowValid(row)) {
                    continue;
                }

                const ui64 yandexuid = FromString<ui64>(row["yandexuid"].AsString());
                const auto& homeCoordinates = row["predicted_home"];
                const auto latitude = homeCoordinates["latitude"].AsDouble();
                const auto longitude = homeCoordinates["longitude"].AsDouble();
                const auto& square = computeSquare({.Lat = latitude, .Lon = longitude}, State->radius());

                /**
                 *    + + -
                 *    + + -
                 *    + - -
                 */
                for (int beltOffset : {-1, 0, 1}) {
                    for (int sqOffset : {-1, 0}) {
                        if (beltOffset == -1 && sqOffset == 0) {
                            continue;
                        }
                        const ui64 square_idx = ConvertSquareToIdx({.Belt = square.Belt + beltOffset, .Sq = square.Sq + sqOffset});
                        TGeoSquare out;
                        out.set_yandexuid(yandexuid);
                        out.set_lat(latitude);
                        out.set_lon(longitude);
                        out.set_squareidx(square_idx);
                        output->AddRow(out);
                    }
                }
            }
        }

        void Save(IOutputStream& output) const override {
            State.Save(output);
        }

        void Load(IInputStream& input) override {
            State.Load(input);
        }

    private:
        TProtoState<TRadius> State;
    };

    class TFindNeighbors: public IReducer<TTableReader<TGeoSquare>, TTableWriter<TNeighborsDistance>> {
    public:
        TFindNeighbors()
                : State()
        {
        }

        TFindNeighbors(const TBuffer& buffer)
                : State(buffer)
        {
        }

        void Do(TTableReader<TGeoSquare>* input, TTableWriter<TNeighborsDistance>* output) override {
            const double radius = State->radius();
            TVector<TGeoSquare> candidates;
            for (; input->IsValid(); input->Next()) {
                const auto& row = input->GetRow();
                candidates.push_back(row);
            }
            for (auto i : xrange(candidates.size())) {
                for (auto j : xrange(i + 1, candidates.size())) {
                    const auto& left = candidates.at(i);
                    const auto& right = candidates.at(j);
                    if (left.yandexuid() == right.yandexuid()) {
                        continue;
                    }
                    double distance = computeDistance({.Lat = left.lat(), .Lon  = left.lon()}, {.Lat = right.lat(), .Lon = right.lon()});
                    if (distance > radius) {
                        continue;
                    }
                    TNeighborsDistance out;
                    out.set_distance(distance);
                    out.set_yandexuidleft(Min(left.yandexuid(), right.yandexuid()));
                    out.set_yandexuidright(Max(left.yandexuid(), right.yandexuid()));
                    output->AddRow(out);
                }
            }
        }

        void Save(IOutputStream& output) const override {
            State.Save(output);
        }

        void Load(IInputStream& input) override {
            State.Load(input);
        }

    private:
        TProtoState<TRadius> State;
    };

    class TUniqueNeighbors: public IReducer<TTableReader<TNeighborsDistance>, TTableWriter<TNeighborsDistance>> {
    public:
        void Do(TTableReader<TNeighborsDistance>* input, TTableWriter<TNeighborsDistance>* output) override {
            for (; input->IsValid(); input->Next()) {
                const auto& row = input->GetRow();
                TNeighborsDistance out;
                out.set_distance(row.distance());
                out.set_yandexuidleft(row.yandexuidleft());
                out.set_yandexuidright(row.yandexuidright());
                output->AddRow(out);
                break;
            }
        }
    };
};

CYT_REGISTER_MAPREDUCER(NFuzzyGeo::TSpaceSlicer, NFuzzyGeo::TFindNeighbors);
CYT_REGISTER_REDUCER(NFuzzyGeo::TUniqueNeighbors);
