#include "geo_utils.h"
#include <cmath>

using namespace NFuzzyGeo;

double degreesToRadians(double degree) {
    return M_PI * degree / 180;
}

double radiansToDegrees(double radians) {
    return radians * 180 / M_PI;
}

double NFuzzyGeo::computeDistance(const TPoint& src, const TPoint& dst) {
    double dLat = degreesToRadians(src.Lat - dst.Lat);
    double dLon = degreesToRadians(src.Lon - dst.Lon);
    double radSrcLat = degreesToRadians(src.Lat);
    double radDstLat = degreesToRadians(dst.Lat);
    double alpha = pow(sin(dLat / 2), 2) + cos(radSrcLat) * cos(radDstLat) * pow(sin(dLon / 2), 2);
    double angle = 2 * atan2(pow(alpha, .5), pow(1. - alpha, .5));
    return EARTH_RADIUS * angle;
}

double computeLonAlpha(double lat, double sideSz) {
    double subRadius = cos(degreesToRadians(lat)) * EARTH_RADIUS;
    if (subRadius > 0) {
        return radiansToDegrees(sideSz / subRadius);
    }
    return 0.;
}

double computeLatAlpha(double sideSz) {
    return radiansToDegrees(sideSz / EARTH_RADIUS);
}

TSquare NFuzzyGeo::computeSquare(const TPoint& p, double sideSz) {
    double latAlpha = computeLatAlpha(sideSz);
    double lonAlpha = computeLonAlpha(p.Lat, sideSz);
    ui32 belt = static_cast<ui32>(floor((p.Lat + 90.) / latAlpha));
    ui32 sq = (lonAlpha > 0) ? static_cast<ui32>(floor((p.Lon + 180.) / lonAlpha)) : 0;
    return {belt, sq};
}

ui64 NFuzzyGeo::ConvertSquareToIdx(const TSquare &square) {
    ui64 squareIdx = square.Belt;
    squareIdx = squareIdx << 32;
    squareIdx += square.Sq;
    return squareIdx;
}

TSquare NFuzzyGeo::ConvertIdxToSquare(ui64 squareIdx) {
    return {static_cast<ui32>(squareIdx >> 32), static_cast<ui32>(squareIdx)};
}
