from crypta.graph.fuzzy.lib.luiger import DateParameter, BaseTask
from crypta.graph.fuzzy.lib.common import cached_property
from crypta.graph.fuzzy.lib.common.dates import days_range_back
from crypta.lib.python.native_yt import run_native_reduce
import crypta.graph.fuzzy.lib.config as conf
from .daily_task import ImportSsidMobileMetrikaDayTask


class ImportSsidMobileMetrikaTask(BaseTask):
    date = DateParameter()
    SSID_THRESHOLD = 20
    YUID_THRESHOLD = 20
    DAYS_IN_MONTH = 7

    def requires(self):
        """
        this tasks must be done to complete this task
        """
        task_list = [
            ImportSsidMobileMetrikaDayTask(date=self.date, target_date=target_date, ssid_threshold=self.SSID_THRESHOLD)
            for target_date in days_range_back(self.date, self.DAYS_IN_MONTH)
        ]
        return task_list

    @property
    def source_type(self):
        return conf.SourceTypes.SSID

    @property
    def source(self):
        return conf.Paths.INDEVICE_PERFECT_NO_LIMIT

    @property
    def destination(self):
        return conf.Paths.sources.ssid.CANDIDATES

    @property
    def sources_schema(self):
        return conf.Paths.sources.ssid.DAY_TABLE_SCHEMA

    @property
    def destination_schema(self):
        return conf.Paths.sources.ssid.CANDIDATES_SCHEMA

    def output(self):
        """
        result of this task
        """
        return [self.yt.targets.table_is_actual(self.destination, self.date.isoformat())]

    @cached_property
    def sources(self):
        return [task.destination for task in self.requires()]

    def query(self, destination):
        sources = ", ".join("`" + source + "`" for source in self.sources)
        return self.render_resource(
            "yql/export_ssid_yuids",
            sources=sources,
            source_nolimit_table=self.yql.path(self.source),
            destination=self.yql.path(destination),
            yuid_threshold=self.YUID_THRESHOLD,
        )

    def _run(self):
        self.yt.create_table_with_schema(
            self.destination, self.destination_schema, strict=True, recreate_if_exists=True
        )
        with self.yt.TempTable() as unexploded, self.yt.TempTable() as not_unique:
            self.yql.execute(self.query(unexploded), syntax_version=1)
            run_native_reduce(
                reducer_name="NCommonWifiAP::TExploder",
                source=unexploded,
                destination=not_unique,
                proxy=self.yt.proxy,
                transaction=self.yt.transaction_id,
                pool=conf.Yt.POOL,
                title="Explode yandexuids with common wifi access point",
                reduce_by=["ssid"],
            )
            yuid_pair = [conf.Constants.YUID_LEFT, conf.Constants.YUID_RIGHT]
            self.yt.run_sort(not_unique, not_unique, sort_by=yuid_pair)
            run_native_reduce(
                reducer_name="NCommonWifiAP::TUnique",
                source=not_unique,
                destination=self.destination,
                proxy=self.yt.proxy,
                transaction=self.yt.transaction_id,
                pool=conf.Yt.POOL,
                title="Make yandexuids with common wifi access point unique",
                reduce_by=yuid_pair,
            )
            self.yt.run_sort(self.destination, sort_by=yuid_pair)
        self.yt.set(self.destination + "/@generate_date", self.date.isoformat())
