#pragma once

#include <crypta/graph/fuzzy/lib/tasks/sources/proto/visitlog_logins_messages.pb.h>
#include <crypta/lib/python/native_yt/cpp/registrar.h>
#include <crypta/lib/python/native_yt/cpp/proto.h>
#include <library/cpp/string_utils/url/url.h>
#include <util/generic/vector.h>
#include <util/generic/set.h>
#include <util/stream/str.h>
#include <util/string/cast.h>
#include <util/generic/maybe.h>
#include <library/cpp/json/json_reader.h>
#include <contrib/libs/re2/re2/re2.h>
#include <util/string/split.h>

using NYT::TNode;
using NYT::IReducer;
using NYT::IMapper;
using NYT::TTableReader;
using NYT::TTableWriter;
using NNativeYT::TProtoState;

template <>
struct TLess<NCommonLogin::TLoginEvent> {
    bool operator()(const NCommonLogin::TLoginEvent& lhs, const NCommonLogin::TLoginEvent& rhs) const {
        const auto lhsTie = std::make_tuple(lhs.GetYandexuid(), lhs.GetLogin(), lhs.GetHost(), lhs.GetKeyword());
        const auto rhsTie = std::make_tuple(rhs.GetYandexuid(), rhs.GetLogin(), rhs.GetHost(), rhs.GetKeyword());
        return lhsTie < rhsTie;
    }
};

namespace NCommonLogin {

    const TString HOST = "host";
    const TString KEYWORD = "keyword";
    const TString LOGIN = "login";
    const TString YANDEXUID = "yandexuid";
    const TString YANDEXUID_LEFT = "yandexuid_left";
    const TString YANDEXUID_RIGHT = "yandexuid_right";
    const TString MATCHED = "matched";
    const TString MATCHED_HOSTS = "matched_hosts";

    class TFilterKeys: public IMapper<TTableReader<TNode>, TTableWriter<TLoginEvent>> {
    public:

        TFilterKeys(): State() {}

        TFilterKeys(const TBuffer& buffer): State(buffer) {}

        void Save(IOutputStream& output) const override {
            State.Save(output);
        }

        void Load(IInputStream& input) override {
            State.Load(input);
        }

        void Do(TReader* input, TWriter* output) override {
            TSet<TString> keywords;
            for (const auto& keyword : State->GetKeywords()) {
                keywords.insert(keyword);
            }
            for (; input->IsValid(); input->Next()) {
                const auto& row = input->GetRow();

                const TString& host = TString{GetOnlyHost(row[URL].AsString())};
                if (IsStopHost(host)) {
                    continue;
                }

                const TString& yuid = row[YUID].AsString();
                const TMaybe<ui64> yandexuid = FromString<ui64>(yuid);
                if (!yandexuid.Defined()) {
                    continue;
                }

                auto& hitParams = row[PARAMS].AsString();
                TVector<TString> hitRows;
                StringSplitter(hitParams).Split('\n').AddTo(&hitRows);
                for (const auto& hitRow : hitRows) {
                    TString timestamps, params;
                    RE2::FullMatch(hitRow, PATTERN, &timestamps, &params);
                    NJson::TJsonValue json = GetJson(params);
                    YieldParams(output, json, 0, yandexuid.GetRef(), host, keywords);
                }
            }
        }

    private:

        NJson::TJsonValue GetJson(const TString& str) {
              TStringStream in;
              in << str;
              NJson::TJsonValue jsonParams;
              NJson::ReadJsonTree(&in, &jsonParams);
              return jsonParams;
        }

        bool IsStopHost(const TString& host) {
            if (host == "" || host == "yadi.sk") {
                return true;
            }
            if (host.Contains("yandex", 0) || host.Contains("kinopoisk")) {
                return true;
            }
            return false;
        }

        static void RemoveSpecialChars(TString& s) {
            TString tmp;
            tmp.reserve(s.size());
            for (auto c = s.begin(); c < s.end(); ++c) {
                if (strchr("-_", *c)) {
                    continue;
                } else {
                    tmp.append(*c);
                }
            }
            s.swap(tmp);
        }

        static TString NormalizeKeyword(const TString& keyword) {
            TString normailized = keyword;
            RemoveSpecialChars(normailized);
            normailized.to_lower();
            return normailized;
        }

        void YieldParams(TWriter* output, const NJson::TJsonValue& json, ui32 level, const ui64 yandexuid, const TString& host, const TSet<TString>& keywords) {
            for (const auto& item : json.GetMap()) {
                const auto& key = NormalizeKeyword(item.first);
                const auto& value = item.second;
                switch (value.GetType()) {
                    case NJson::JSON_MAP: {
                        YieldParams(output, value, level + 1, yandexuid, host, keywords);
                        break;
                    }
                    case NJson::JSON_ARRAY:
                    case NJson::JSON_UNDEFINED:
                    case NJson::JSON_NULL: {
                        break;
                    }
                    default: {
                        if (keywords.contains(key)) {
                            TLoginEvent params;
                            params.SetHost(host);
                            params.SetYandexuid(yandexuid);
                            params.SetKeyword(key);
                            params.SetLogin(value.GetStringRobust());
                            output->AddRow(params);
                        }
                        break;
                    }
                }
            }
        }

        const TString PATTERN = "^[[](.+)+[]][[:space:]]+(.+)$";
        const TString PARAMS = "Params";
        const TString YUID = "UserID";
        const TString URL = "StartURL";
        TProtoState<TFilterKeysOptions> State;
    };

    class TFilterRareLogins: public IReducer<TTableReader<TLoginEvent>, TTableWriter<TLoginEvent>> {
    public:

        TFilterRareLogins(): State() {}

        TFilterRareLogins(const TBuffer& buffer): State(buffer) {}

        void Do(TReader* input, TWriter* output)  override {
            TSet<ui64> yandexuids;
            TSet<TLoginEvent> loginRecords;
            const ui32 threshold = State->GetThreshold();
            for (; input->IsValid(); input->Next()) {
                const auto& row = input->GetRow();
                yandexuids.insert(row.GetYandexuid());
                if (yandexuids.size() > threshold) {
                    return;
                }
                loginRecords.insert(row);
            }
            for (const auto& record : loginRecords) {
                output->AddRow(record);
            }
        }

        void Save(IOutputStream& output) const override {
            State.Save(output);
        }

        void Load(IInputStream& input) override {
            State.Load(input);
        }

    private:
        TProtoState<TFilterRareLoginsOptions> State;
    };

    class TExpose: public IReducer<TTableReader<TLoginEvent>, TTableWriter<TSimpleMatch>> {
    public:

        TExpose(): State() {}

        TExpose(const TBuffer& buffer): State(buffer) {}

        void Save(IOutputStream& output) const override {
            State.Save(output);
        }

        void Load(IInputStream& input) override {
            State.Load(input);
        }

        void Do(TReader* input, TWriter* output)  override {
            TSet<ui64> yandexuids;
            bool is_first_row = true;
            TSimpleMatch match;
            const ui32 threshold = State->GetThreshold();
            for (; input->IsValid(); input->Next()) {
                const auto& row = input->GetRow();
                if (is_first_row) {
                    is_first_row = false;
                    match.SetLogin(row.GetLogin());
                    match.SetKeyword(row.GetKeyword());
                    match.SetHost(row.GetHost());
                }
                yandexuids.insert(row.GetYandexuid());
                if (yandexuids.size() > threshold) {
                    return;
                }
            }
            if (yandexuids.size() < 2) {
                return;
            }
            for (auto i = yandexuids.begin(); i != yandexuids.end(); ++i) {
                for (auto j = std::next(i); j != yandexuids.end(); ++j) {
                    match.SetYandexuidLeft(Min(*i, *j));
                    match.SetYandexuidRight(Max(*i, *j));
                    output->AddRow(match);
                }
            }
        }
    private:
        TProtoState<TFilterRareLoginsOptions> State;
    };

    class TUnique: public IReducer<TTableReader<TSimpleMatch>, TTableWriter<TMatch>> {
    public:
        void Do(TReader* input, TWriter* output)  override {
            TSet<TString> hosts;
            bool is_first_row = true;
            TMatch match;
            for (; input->IsValid(); input->Next()) {
                const auto& row = input->GetRow();
                if (is_first_row) {
                    is_first_row = false;
                    match.SetYandexuidLeft(row.GetYandexuidLeft());
                    match.SetYandexuidRight(row.GetYandexuidRight());
                }
                const auto& host = row.GetHost();
                hosts.insert(host);
                /*
                auto matchRecord = match.AddMatched();
                matchRecord->SetHost(host);
                matchRecord->SetKeyword(row.GetKeyword());
                matchRecord->SetLogin(row.GetLogin());
                */
            }
            match.SetMatchedHosts(hosts.size());
            output->AddRow(match);
        }
    };
}

CYT_REGISTER_MAPREDUCER(NCommonLogin::TFilterKeys, NCommonLogin::TFilterRareLogins);
CYT_REGISTER_REDUCER(NCommonLogin::TExpose);
CYT_REGISTER_REDUCER(NCommonLogin::TUnique);
