"""
Dinanzi a me non fuor cose create
se non etterne, e io etterna duro.
Lasciate ogni speranza, voi ch'entrate.
"""

import base64
import math
from collections import namedtuple

from yt import yson


class main_profile_config(object):
    """ incapsulate main_profile_config into scope """
    __slots__ = ()

    mapping_index_size = 1000

    class TSocdemPriority(object):
        # sex : 0 - male, 1 - female
        # age : 0 - <18, 1 - 18-24, 2 - 25-34, 3 - 35-44, 4 - 45-54, 5 - >=54

        __slots__ = ('age', 'sex', )

        def __init__(self, sex=None, age=None):
            self.sex = sex
            self.age = age

    priorities_0_999 = [
        TSocdemPriority(1, 3),
        TSocdemPriority(1, 2),
        TSocdemPriority(1, 4),
        TSocdemPriority(0, 3),
        TSocdemPriority(0, 2),
        TSocdemPriority(0, 4),
        TSocdemPriority(1, 1),
        TSocdemPriority(0, 1),
        TSocdemPriority(1, 5),
        TSocdemPriority(0, 5),
        TSocdemPriority(0, 0),
        TSocdemPriority(1, 0)
    ]

    sex_priority_0_999 = 1  # female
    socdem_priorities = [(999, priorities_0_999, sex_priority_0_999)]

    YuidCryptaIdPair = namedtuple('YuidCryptaIdPair', ['yuid', 'crypta_id'])


def get_field_value(name, tskv):
    for item in tskv.split('\t'):
        pair = item.split('=', 1)
        if len(pair) == 2 and pair[0] == name:
            return pair[1]
    return ''


class YqlMkqlStruct(object):

    """ Allow to safe get data from Yql Mkql struct by call get() or [] """

    YSON_FIELDS = ('exact_socdem', 'income_5_segments', 'probabilistic_segments', 'predicted_home', )
    __slots__ = ('struct', )

    def __init__(self, struct):
        self.struct = struct
        self._load_yson()

    def get(self, key, default=None):
        return getattr(self.struct, key, default)

    def __getitem__(self, key):
        return self.get(key)

    def _load_yson(self):
        for field in self.YSON_FIELDS:
            parsed = {}
            if getattr(self.struct, field, None) is not None:
                parsed = yson.loads(getattr(self.struct, field))
            setattr(self.struct, field, parsed)


class HouseholdInfo(object):

    attr_meta = [
        # order_idx, name, number_of_bits
        (1, 'version', 2),
        (2, 'male', 1),
        (3, 'female', 1),
        (4, 'child', 1),
        (5, 'grand', 1),
        (6, 'inc', 2),
        (7, 'cbrsrs', 4),
        (8, 'cindiv', 4),
        (9, 'ccoll', 4),
        (10, 'cstat', 4),
        (11, 'ctrav', 4),
        (12, 'desk', 1),
        (13, 'mobapp', 1),
        (14, 'mobweb', 1),
        (15, 'smart', 1),
        (16, 'otherp', 1),
        (17, 'win', 1),
        (18, 'ios', 1),
        (19, 'android', 1),
        (20, 'macos', 1),
        (21, 'otheros', 1),
        # hh geo
        (22, 'lat_sign', 1),
        (23, 'lat', 25),
        (24, 'lon_sign', 1),
        (25, 'lon', 25),
        # main profile attributes (see CRYPTAIS-400)
        (26, 'mp_male', 1),
        (27, 'mp_female', 1),
        (28, 'mp_l18', 1),
        (29, 'mp_18_24', 1),
        (30, 'mp_25_34', 1),
        (31, 'mp_35_44', 1),
        (32, 'mp_45_54', 1),
        (33, 'mp_g55', 1),
        (34, 'mp_align_bit_no_meaning', 1),
        # adhocs
        (35, 'cadhocs', 12),
        (36, 'c3dpartydata', 12),
        (37, 'adhocs', -1),
        (38, '3dpartydata', -1)
    ]

    def __init__(self, hh_data):
        self.hh_id = hh_data.get('hh_id', 0)

        self.attrs = {}
        for attr in self.attr_meta:
            self.attrs[attr[1]] = 0

        self.attrs['version'] = 0
        self.genders = hh_data.get('sex', [])
        # [[0 - male], [1 - female]]
        self.attrs['male'] = 1 if 0 in self.genders else 0
        self.attrs['female'] = 1 if 1 in self.genders else 0

        self.ages = set(hh_data.get('age', []))
        self.attrs['child'] = 1 if 0 in self.ages else 0
        self.attrs['grand'] = 1 if 5 in self.ages else 0

        self.inc = hh_data.get('inc')
        self.attrs['inc'] = self.inc + 1 if self.inc is not None else 0

        self.brsrs = set(hh_data.get('brs', []))
        self.attrs['cbrsrs'] = min(len(self.brsrs), 15)

        self.yc = hh_data['yc']
        self.dc = hh_data['dc']
        self.oss = set(operation_system.lower() for operation_system in hh_data['oss'])
        self.pltfrms = hh_data['pltfrms']

        # fast fix; TODO: change hh bb format to support adhoc-ids with more than 12 bit length
        self.adhocs = [adhc for adhc in hh_data['adhocs'] if adhc < (1 << 12)]
        self.thirdpty = []

        if 'd' in self.pltfrms:
            self.attrs['desk'] = 1
        if self.dc > 0:
            self.attrs['mobapp'] = 1
        if 'm' in self.pltfrms:
            self.attrs['mobweb'] = 1

        if 'windows' in self.oss:
            self.attrs['win'] = 1
        if 'ios' in self.oss:
            self.attrs['ios'] = 1
        if 'android' in self.oss:
            self.attrs['android'] = 1
        if 'macos' in self.oss:
            self.attrs['macos'] = 1
        if len(self.oss - {'windows', 'ios', 'android', 'macos'}) > 0:
            self.attrs['otheros'] = 1

        self.attrs['adhocs'] = self.adhocs
        self.attrs['cadhocs'] = len(self.adhocs)

        self.attrs['3dpartydata'] = self.thirdpty
        self.attrs['c3dpartydata'] = len(self.thirdpty)

        # geo
        lat = hh_data.get('lat', 0.)
        self.attrs['lat_sign'] = 0 if lat >= 0 else 1
        self.attrs['lat'] = int(math.fabs(lat) * (1 << 17))

        lon = hh_data.get('lon', 0.)
        self.attrs['lon_sign'] = 0 if lon >= 0 else 1
        self.attrs['lon'] = int(math.fabs(lon) * (1 << 17))

        # main profile socdem flags
        mp_sex = hh_data['mp_sex']
        mp_age = hh_data['mp_age']
        mp_sex_bits = [(1 if ind in mp_sex else 0) for ind in xrange(2)]
        mp_age_bits = [(1 if ind in mp_age else 0) for ind in xrange(6)]
        self.attrs['mp_male'], self.attrs['mp_female'] = mp_sex_bits
        (
            self.attrs['mp_l18'], self.attrs['mp_18_24'],
            self.attrs['mp_25_34'], self.attrs['mp_35_44'],
            self.attrs['mp_45_54'], self.attrs['mp_g55']
        ) = mp_age_bits

        # bit added specially for pshevtsov@
        self.attrs['mp_align_bit_no_meaning'] = 0

        self.bitstring = self.make_bitstring()

    def make_bitstring(self):
        res = ''
        for attr in self.attr_meta:
            attr_code = attr[1]
            attr_len = attr[2]
            if attr_len != -1:
                res += ('{:0%db}' % attr_len).format(self.attrs[attr_code])
            else:
                for a in self.attrs[attr_code]:
                    a_bin = '{:012b}'.format(int(a)).strip()
                    if len(a_bin) > 12:
                        # raise ValueError('Too big attr_value: ' + a)
                        continue
                    res += a_bin

        # remove trailing zero bytes
        res.rstrip('0')

        # add zeroes for full byte at the end
        res += '0' * ((8 - len(res) % 8) % 8)
        return res

    def id(self):
        return self.hh_id

    def string(self):
        return str(bytearray(self.bytes()))

    def base64(self):
        return base64.standard_b64encode(self.string())

    def base64_bytes(self):
        return [b for b in bytearray(self.base64())]

    def base64_binary(self, bytes_separator=''):
        res = []
        for b in self.base64_bytes():
            res.append('{:08b}'.format(b))
        return bytes_separator.join(res)

    def bytes(self):
        bs = []
        s = '{:032b}'.format(self.hh_id) + self.bitstring
        for i in range(0, len(s), 8):
            bs.append(int(s[i:i + 8], 2))
        return bs

    def tns(self):
        s = self.bitstring
        return s[2:6] + {'00': '0', '01': 'A', '10': 'B', '11': 'C'}[s[6:8]]


class HhComposition(object):

    """ Collect hh info """

    __slots__ = (
        'hh_data', 'hh_info', 'household_id', 'household_id_32',
        'yuid_set', 'crypta_id_set', 'device_set', 'size', 'diff_socdems',
        'hh_age', 'hh_sex', 'hh_inc', 'all_adhocs', 'socdem_profiles', 'smart_tv_yuids',
        'main_profile_sex', 'main_profile_age', 'main_profile_yuid',
    )

    SEX_TO_INT = {'m': 0, 'f': 1}
    AGE_TO_INT = {'0_17': 0, '18_24': 1, '25_34': 2, '35_44': 3, '45_54': 4, '55_99': 5}
    INC_TO_INT = {'A': 0, 'B': 1, 'C': 2}

    def __init__(self, household_id, household_id_32):
        self.household_id = int(household_id)
        self.household_id_32 = household_id_32
        self.hh_data = dict()
        self.hh_data['brs'] = set()
        self.hh_data['oss'] = set()
        self.hh_data['pltfrms'] = set()

        self.yuid_set = set()
        self.device_set = set()
        self.crypta_id_set = set()

        self.hh_age = set()
        self.hh_sex = set()
        self.hh_inc = {'A': 0., 'B': 0., 'C': 0.}
        self.socdem_profiles = []    # format : [((male_p, female_p), (age_0, ..., age_5), [adhoc, ...]), ...]
        self.smart_tv_yuids = []

        self.size = 0
        self.diff_socdems = set()

    def add_yuid(self, uniq_id, record):
        """
        Add new yuid to hh data
        uniq_id - is tuple YuidCryptaIdPair(yuid, crypta_id)
        """
        self.size += 1
        self.yuid_set.add(uniq_id.yuid)
        self.crypta_id_set.add(uniq_id.crypta_id)

        income = record.get('income_5_segments')
        if income is not None:
            # join income to households
            self.hh_inc['A'] += income.get('A', 0.)
            self.hh_inc['B'] += income.get('B1', 0.) + income.get('B2', 0.) + income.get('B', 0.)
            self.hh_inc['C'] += income.get('C1', 0.) + income.get('C2', 0.) + income.get('C', 0.)

        age = self.AGE_TO_INT.get(self._get_record_data(record, 'exact_socdem', 'age_segment'))
        if age is not None:
            # unpdate age from cookie
            self.hh_age.add(age)

        # The check 18-45 for sex is requested here:
        # https://wiki.yandex-team.ru/users/sirina0/notes/advideoproduct/advideoproductdetails/HouseHoldProduct#h-1
        sex = self.SEX_TO_INT.get(self._get_record_data(record, 'exact_socdem', 'gender'))
        if (sex is not None) and (age not in {0, 5}):
            # unpdate sex from cookie
            self.hh_sex.add(sex)

        if age is not None or sex is not None:
            self.diff_socdems.add((age, sex))
        adhocs = list(map(int, (record.get('probabilistic_segments') or {}).keys()))
        self.socdem_profiles.append((sex, age, adhocs, uniq_id))

        # extra info from yuid-with-all
        # ua_profile is 'd|desk|windows|10.0' or some of 'd|...' or 'm|...'
        # at yql side we split to we split yuid_wit_all.ua_profile_os > to ua_profile_os and ua_profile_platform
        # and add yuid_wit_all.browser as ua_profile_browser
        if record.get('ua_profile_browser'):
            self.hh_data['brs'].add(record.get('ua_profile_browser'))
        if record.get('ua_profile_os'):
            self.hh_data['oss'].add(record.get('ua_profile_os'))
        if record.get('ua_profile_platform'):
            self.hh_data['pltfrms'].add(record.get('ua_profile_platform'))

        if record.get('predicted_home'):
            self.hh_data['lat'] = record['predicted_home']['latitude']
            self.hh_data['lon'] = record['predicted_home']['longitude']

    def add_device(self, device_id, record):
        """ Add new device to hh data """
        # TODO: make some with device
        self.device_set.add(device_id)

        if record.get('os'):
            self.hh_data['oss'].add(record['os'])

        self.hh_data['pltfrms'].add('a')

    def iterate_records(self, records):
        """ Loop over all records """
        for yql_record in records:
            # if yuid? (always yuid!)
            record = YqlMkqlStruct(yql_record)
            self.add_yuid(main_profile_config.YuidCryptaIdPair(
                record.get('yuid'), record.get('crypta_id')), record)

    def find_main_profile(self):
        """ Try to find yuid with max priority and get common household socdem """
        # get correct priority list using hash of hh_id
        map_index = (self.household_id * 19 * 31) % 11719  # some simple hash in [0:10000]
        map_index = map_index % main_profile_config.mapping_index_size
        priorities, sex_priority = [], None
        for index, conf_pr, conf_sex_pr in main_profile_config.socdem_priorities:
            if map_index <= index:
                priorities = conf_pr
                sex_priority = conf_sex_pr
                break

        # find yuids with max priority
        adhocs_canditats = []
        max_priority = None
        for sex, age, adhocs, yuid in self.socdem_profiles:
            for index, pr in enumerate(priorities):
                if (pr.sex == sex) and (pr.age == age):
                    if max_priority is None or index < max_priority:
                        max_priority = index
                        adhocs_canditats = [(adhocs, yuid)]
                    elif index == max_priority:
                        adhocs_canditats.append((adhocs, yuid))

        all_adhocs = []
        if max_priority is not None:
            # has yuid with main profile
            self.main_profile_sex = priorities[max_priority].sex
            self.main_profile_age = priorities[max_priority].age
            all_adhocs, self.main_profile_yuid = max(
                adhocs_canditats, key=lambda item: len(item[0])
            )
        else:
            self.main_profile_yuid = main_profile_config.YuidCryptaIdPair(None, None)
            # no main yuid, so no adhocs, only set main socdem flags
            if sex_priority in self.hh_sex:  # probably 1
                self.main_profile_sex = sex_priority
            elif (1 - sex_priority) in self.hh_sex:  # probably 0
                self.main_profile_sex = 1 - sex_priority
            else:
                self.main_profile_sex = None

            if 0 in self.hh_sex or 1 in self.hh_sex:
                self.main_profile_age = 2
            elif 5 in self.hh_age:
                self.main_profile_age = 5
            elif 0 in self.hh_age:
                self.main_profile_age = 0
            else:
                self.main_profile_age = None

        self.all_adhocs = set(all_adhocs)

    def finalize_data(self):
        """ Make final hh data preparing """
        # CRYPTAIS-226
        max_inc = max(self.hh_inc.items(), key=lambda pair: pair[1])
        if max_inc[1] > 0:
            max_inc = self.INC_TO_INT.get(max_inc[0])
        else:
            max_inc = None

        self.hh_data.update(
            hh_id=self.household_id_32,
            sex=self.hh_sex,
            age=self.hh_age,
            inc=max_inc,
            yuids=self.yuid_set,
            devids=self.device_set,
            crypta_ids=self.crypta_id_set,
            yc=len(self.yuid_set),    # ?
            dc=len(self.device_set),  # ?
            mp_sex=set([self.main_profile_sex]) if self.main_profile_sex is not None else set(),
            mp_age=set([self.main_profile_age]) if self.main_profile_age is not None else set(),
            adhocs=list(sorted(self.all_adhocs))
        )
        self.hh_info = HouseholdInfo(self.hh_data)

    def dump(self):
        """ Save household info into dict """
        return {
            'hhid': str(self.household_id),
            'hhid_32': self.household_id_32,
            'data': self._dump(self.hh_data),
            'size': self.size,
            'socdems': len(self.diff_socdems),
            'info_binary': self.hh_info.base64_binary(),
            'info': self._dump(self.hh_info.attrs),
            'main_profile_yuid': self.main_profile_yuid.yuid,
            'main_profile_crypta_id': self.main_profile_yuid.crypta_id,
        }

    def _get_record_data(self, record, *keys):
        for key in keys:
            record = record.get(key, {})
            if record is None:
                break
        return record or None

    def _dump(self, dict_data):
        for key in dict_data.keys():
            if isinstance(dict_data[key], set):
                dict_data[key] = sorted(list(dict_data[key]))
        return yson.dumps(dict_data)


def hh_composition_reducer((hhid, hhid_32), rows):
    composition = HhComposition(hhid, hhid_32)
    composition.iterate_records(rows)
    composition.find_main_profile()
    composition.finalize_data()
    yield composition.dump()
