from __future__ import print_function

import base64
from collections import OrderedDict


class HouseholdInfoDecoder(object):

    attr_meta = [
        # order_idx, name, number_of_bits
        (0, 'hh_id', 32),
        (1, 'version', 2),
        (2, 'male', 1),
        (3, 'female', 1),
        (4, 'child', 1),
        (5, 'grand', 1),
        (6, 'inc', 2),
        (7, 'cbrsrs', 4),
        (8, 'cindiv', 4),
        (9, 'ccoll', 4),
        (10, 'cstat', 4),
        (11, 'ctrav', 4),
        (12, 'desk', 1),
        (13, 'mobapp', 1),
        (14, 'mobweb', 1),
        (15, 'smart', 1),
        (16, 'otherp', 1),
        (17, 'win', 1),
        (18, 'ios', 1),
        (19, 'android', 1),
        (20, 'macos', 1),
        (21, 'otheros', 1),
        # hh geo
        (22, 'lat_sign', 1),
        (23, 'lat', 25),
        (24, 'lon_sign', 1),
        (25, 'lon', 25),
        # main profile attributes (see CRYPTAIS-400)
        (26, 'mp_male', 1),
        (27, 'mp_female', 1),
        (28, 'mp_l18', 1),
        (29, 'mp_18_24', 1),
        (30, 'mp_25_34', 1),
        (31, 'mp_35_44', 1),
        (32, 'mp_45_54', 1),
        (33, 'mp_g55', 1),
        (34, 'mp_align_bit_no_meaning', 1),
        # adhocs
        (35, 'cadhocs', 12),
        (36, 'c3dpartydata', 12),
        (37, 'adhocs', -1),
        (38, '3dpartydata', -1)
    ]

    def decode(self, b64_string):
        data = base64.b64decode(b64_string)
        binary = self._string_to_bin(data)
        attr = self.dict_from_bitstring(binary)
        return attr

    def decode_bin(self, base64_binary_str):
        bytes = self._bin_to_bytes(base64_binary_str)
        return self.decode(self._byte_to_string(bytes))

    def dict_from_bitstring(self, bitstring):
        """  """
        result = OrderedDict()
        offset = 0
        for (_, key, bitlen) in self.attr_meta:
            if bitlen != -1:
                result[key] = int(bitstring[offset:offset + bitlen], 2)
                offset += bitlen
            else:
                count = result['c{0}'.format(key)]
                d_bitlen = 12
                data = []
                for index in range(count):
                    data.append(int(bitstring[offset:offset + d_bitlen], 2))
                    offset += d_bitlen
                result[key] = data
        return result

    def _bin_to_bytes(self, binary_str):
        """ Convert 0, 1 binary string into bytes list """
        bytes = []
        for index in range(0, len(binary_str), 8):
            bytes.append(int(binary_str[index:index + 8], 2))
        return bytes

    def _byte_to_string(self, byte_list):
        """ Convert list of bytes into ASCII string """
        return ''.join(map(chr, byte_list))

    def _string_to_bin(self, byte_string):
        """ Convert byte string to bit string """
        return ''.join(map('{:08b}'.format, map(ord, byte_string)))


if __name__ == '__main__':

    def main():
        import sys
        decoder = HouseholdInfoDecoder()
        data = sys.argv[1]
        for line in (decoder.decode(data)).items():
            print(line)

    main()
