from __future__ import print_function
import contextlib
import datetime
import time
import uuid

from cached_property import cached_property
from crypta.lib.python.bt.workflow import IndependentTask
from crypta.lib.python.yql_runner.task import YQLRunnerTask
import crypta.lib.python.bt.conf.conf as conf


class HouseHoldsQuery(YQLRunnerTask, IndependentTask):

    """ Base house hold query class """

    GRAPH_OUTPUT_DIR = "//home/crypta/{crypta_env}/state/graph"
    HOUSEHOLDS_OUTPUT_DIR = "//home/crypta/{crypta_env}/state/households_new"
    QUERY_TEMPLATE = None

    @cached_property
    def date(self):
        return conf.proto.Date

    def get_graph_output_dir(self):
        return self.GRAPH_OUTPUT_DIR.format(crypta_env=self.crypta_env)

    def get_households_base_dir(self):
        return self.HOUSEHOLDS_OUTPUT_DIR.format(crypta_env=self.crypta_env)

    def get_context_data(self, **kwargs):
        context = super(HouseHoldsQuery, self).get_context_data(**kwargs)
        context.update(
            date=self.date, household_dir=self.get_households_base_dir(), graph_output_dir=self.get_graph_output_dir()
        )
        return context

    def get_libs(self):
        return []

    def set_generate_date(self):
        attributes = self.yt.get("{path}/@".format(path=self.get_households_base_dir()))
        output = self.yt.list("{path}/output".format(path=self.get_households_base_dir()), absolute=True)

        for path in output:
            self.yt.set("{path}/@generate_date".format(path=path), attributes.get("generate_date"))
            self.yt.set("{path}/@run_date".format(path=path), attributes.get("run_date"))


class HouseHoldsStream(HouseHoldsQuery):

    """ TODO: use stream from incremental log import """

    PROCESSED_TABLES = "//home/crypta/{crypta_env}/state/households_new/processed"

    @contextlib.contextmanager
    def run_context(self):
        with self.yt.Transaction():
            self.yt.create(
                "map_node", "{0}/stream".format(self.get_households_base_dir()), recursive=True, ignore_existing=True
            )
            self.yt.create(
                "map_node", "{0}/storage".format(self.get_households_base_dir()), recursive=True, ignore_existing=True
            )

        with super(HouseHoldsStream, self).run_context() as ctx:
            yield ctx

        self._set_expiration_time()

    @cached_property
    def uniqid(self):
        return "{time:.0f}-{uuid}".format(time=time.time(), uuid=uuid.uuid4())

    def run(self, *args, **kwargs):
        result = super(HouseHoldsStream, self).run(*args, **kwargs)
        return self._after_run(result)

    def get_processed_tables_storage(self):
        return self.PROCESSED_TABLES.format(crypta_env=self.crypta_env)

    def get_context_data(self, **kwargs):
        context = super(HouseHoldsStream, self).get_context_data(**kwargs)
        context.update(uniqid=self.uniqid, processed_tables=self.get_processed_tables_storage())
        return context

    def _after_run(self, result):
        if result is None:
            return
        output_table, processed_tables = result
        if hasattr(processed_tables, "rows"):
            # yql client
            data = ({"path": path[0]} for path in processed_tables.rows)
        else:
            # embeded yql (in tests)
            assert output_table["Label"] == "result table"
            assert processed_tables["Label"] == "processed tables"
            data = ({"path": path[0]} for path in processed_tables["Write"][0]["Data"])

        self.yt.write_table(self.yt.TablePath(self.get_processed_tables_storage(), append=True), data, format="json")
        return output_table

    def _set_expiration_time(self):
        """ Update ttl for storage tables """
        for path in self.yt.list("{0}/storage".format(self.get_households_base_dir()), absolute=True):
            dt = datetime.datetime.strptime(path.split("/")[-1], "%Y-%m-%d") + datetime.timedelta(days=95)
            self.yt.set("{path}/@expiration_time".format(path=path), "{dt:%Y-%m-%d} 00:00:00.0+00:00".format(dt=dt))
