package ru.yandex.crypta.graph2.dao.yql;

import java.sql.SQLException;
import java.util.Map;
import java.util.function.Consumer;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.graph2.dao.yt.ops.CurrentScriptName;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.yql.YqlConnection;
import ru.yandex.yql.YqlDataSource;
import ru.yandex.yql.YqlStatement;
import ru.yandex.yql.settings.YqlProperties;

public class YqlImpl implements Yql {

    private static final Logger LOG = LoggerFactory.getLogger(YqlImpl.class);
    private final YqlDataSource yqlDataSource;
    private String queryHeader;

    public YqlImpl(String yqlUrl, String yqlYtCluster, String yqlToken, String ytPool) {
        yqlDataSource = new YqlDataSource(
                String.format("jdbc:yql://%s/%s?syntaxVersion=1",
                        yqlUrl,
                        yqlYtCluster
                ),
                new YqlProperties().withCredentials(
                        "not_used",
                        yqlToken
                ));

        this.queryHeader = "";
        if (ytPool != null && !ytPool.isEmpty()) {
            queryHeader += String.format("\nPRAGMA yt.Pool = '%s';\n", ytPool);
        }

    }

    @Override
    public void executeInTransaction(GUID txId, String queryTitle, String queryTemplate, Map<String, String> params) {
        String query = queryTemplate;
        for (String paramKey : params.keySet()) {
            String paramValue = params.get(paramKey);
            query = query.replace("{" + paramKey + "}", paramValue);
        }
        String txPragma = String.format("PRAGMA yt.ExternalTx='%s';\n", txId);
        String fullQuery = this.queryHeader + txPragma + query;
        LOG.info("Executing:\n" + fullQuery);
        executeQuery(fullQuery, queryTitle);
    }

    private void executeQuery(String query, String queryTitle) {
        withStatement(yqlStatement -> {
            try {
                String scriptName = CurrentScriptName.get();
                if (scriptName != null) {
                    yqlStatement.setAdditionalAttributes(Map.of("script_name", scriptName));
                }
                yqlStatement.setTitle(queryTitle);
                yqlStatement.execute(query);
            } catch (SQLException e) {
                throw ExceptionUtils.translate(e);
            }
        });
    }


    private void withStatement(Consumer<YqlStatement> callback) {
        // helper method to properly handle connection using try-with-resources
        try (YqlConnection connection = (YqlConnection) yqlDataSource.getConnection()) {
            try (YqlStatement statement = connection.createYqlStatement()) {
                callback.accept(statement);
            }
        } catch (SQLException e) {
            throw ExceptionUtils.translate(e);
        }
    }
}
