package ru.yandex.crypta.graph2.dao.yt;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.graph2.utils.NativeLibHelper;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtConfiguration;
import ru.yandex.inside.yt.kosher.impl.operations.jars.JarsProcessor;
import ru.yandex.inside.yt.kosher.impl.operations.jars.SingleUploadFromClassPathJarsProcessor;

import static ru.yandex.crypta.graph2.utils.EnvUtils.getEnvOrDefault;
import static ru.yandex.crypta.graph2.utils.EnvUtils.getEnvOrThrow;

public class YtConfig {

    public static final YPath DEFAULT_JARS_DIR = YPath.simple("//tmp/java-yt-wrapper/jars");
    private static final Logger LOG = LoggerFactory.getLogger(YtConfig.class);

    public static String getYtProxy() {
        return getEnvOrDefault("YT_PROXY", "hahn.yt.yandex.net");
    }

    public static String getYtToken() {
        return getEnvOrThrow("YT_TOKEN");
    }

    public static String getYtPool() {
        return getEnvOrThrow("YT_POOL");
    }

    public static String getTmpDir() {
        return getEnvOrDefault("YT_JAVA_TMP_DIR", null);
    }

    public static String getCustomJavaBinary() {
        return getEnvOrDefault("YT_JAVA_BINARY", null);
    }

    /**
     * Initialize YT client from env vars
     *
     * @return YT client
     */
    public static Yt getYt() {
        return getYt(false);
    }

    /**
     * Initialize YT client from env vars
     *
     * @param withNativeLibsSupport upload local .so files to YT (used e.g. for JNI)
     * @return YT client
     */
    public static Yt getYt(boolean withNativeLibsSupport) {

        YtConfiguration.Builder conf = YtConfiguration.builder()
                .withApiHost(YtConfig.getYtProxy())
                .withToken(YtConfig.getYtToken())
                .withPool(YtConfig.getYtPool());

        String customJavaBinary = getCustomJavaBinary();
        if (customJavaBinary != null) {
            LOG.info("Custom java binary is used: {}", customJavaBinary);
            conf = conf.withJavaBinary(customJavaBinary).withJobSpecPatch(null).withSpecPatch(null);
        } else {
            conf = conf.withPortoJava17();
        }

        if (withNativeLibsSupport && !".".equals(NativeLibHelper.getJavaLibraryPath())) {
            LOG.warn("Too many system native libs(.so) may be uploaded to YT when using default {}. " +
                    "Use NativeLibHelper#setLocalJavaLibraryPath to fix it", NativeLibHelper.JAVA_LIBRARY_PATH);
        }

        String tmpDir = getTmpDir();
        JarsProcessor jarsProcessor;
        if (tmpDir != null) {
            // Sometimes tmp dir is cleaned up unexpectedly, this crushes client.
            // This hack is used to move tmp dir to some permanent place.
            YPath tmpPath = YPath.simple(tmpDir);
            YPath jarsDir = tmpPath.child("jars");

            // in fact, YT file cache is used most of the time
            jarsProcessor = new SingleUploadFromClassPathJarsProcessor(
                    jarsDir, withNativeLibsSupport
            );

            conf = conf.withTmpDir(tmpPath);
        } else {
            jarsProcessor = new SingleUploadFromClassPathJarsProcessor(
                    DEFAULT_JARS_DIR, withNativeLibsSupport
            );
        }

        return Yt.builder(conf.build())
                .http()
                .withJarsProcessor(jarsProcessor)
                .build();
    }

}
