package ru.yandex.crypta.graph2.dao.yt;

import java.time.LocalDate;
import java.util.Map;

import com.google.protobuf.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.dao.yt.schema.extractor.ProtobufYtSchemaExtractor;
import ru.yandex.crypta.graph2.dao.yt.schema.extractor.YTreeYtSchemaExtractor;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.RangeLimit;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.inside.yt.kosher.transactions.Transaction;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.yt.ytclient.tables.TableSchema;


public class YtCypressHelper {
    // generate_date name is used for compatibility with v2
    public static final String GENERATION_DATE_ATTR = "generate_date";
    public static final String RUN_DATE_ATTR = "run_date";
    private static final Logger LOG = LoggerFactory.getLogger(YtCypressHelper.class);
    private final Yt yt;
    private final YTreeYtSchemaExtractor ytSchemaExtractor;
    private final ProtobufYtSchemaExtractor protobufYtSchemaExtractor;

    public YtCypressHelper(Yt yt) {
        this.yt = yt;
        this.ytSchemaExtractor = new YTreeYtSchemaExtractor();
        this.protobufYtSchemaExtractor = new ProtobufYtSchemaExtractor();
    }

    public static YPath mkRangeByKey(YPath path, String key) {
        return path.withRange(
                new RangeLimit(Cf.list(YTree.stringNode(key)), -1, -1),
                new RangeLimit(Cf.list(YTree.stringNode(key + '\0')), -1, -1)
        );
    }

    public static YPath mkRangeByKey(YPath path, ListF<String> compoundKey) {
        // extend last key part of compound key with additional symbol to make interval inclusive
        ListF<String> compoundKeySecondInterval = Cf.toArrayList(compoundKey);
        compoundKeySecondInterval.set(compoundKeySecondInterval.size() - 1, compoundKeySecondInterval.last() + '\0');

        RangeLimit from = new RangeLimit(compoundKey.map(YTree::stringNode), -1, -1);
        RangeLimit to = new RangeLimit(compoundKeySecondInterval.map(YTree::stringNode), -1, -1);
        return path.withRange(from, to);

    }

    public ListF<YPath> ls(String path) {
        return ls(YPath.simple(path));
    }

    public ListF<YPath> ls(YPath path) {
        return Cf.wrap(yt.cypress().list(path))
                .map(YTreeNode::stringValue)
                .map(path::child);
    }

    public Option<YPath> checkExistence(YPath p) {
        if (yt.cypress().exists(p)) {
            return Option.of(p);
        } else {
            return Option.empty();
        }
    }

    public long rowCount(YPath table) {
        return yt.cypress().get(table.attribute("row_count")).longValue();
    }

    public boolean isEmpty(YPath table) {
        return rowCount(table) == 0;
    }

    public void ensureDir(YPath table) {
        LOG.info("Ensuring dir: {}", table);
        yt.cypress().create(table, CypressNodeType.MAP, true, true);
    }

    public <T> TableSchema extractTableSchema(Class<T> clazz) {
        ClassX<?> classX = ClassX.wrap(clazz);
        if (classX.hasAnnotationInAnnotatedOrItsAnnotations(YTreeObject.class)) {
            return ytSchemaExtractor.extractTableSchema(clazz);
        } else if (Message.class.isAssignableFrom(clazz)) {
            return protobufYtSchemaExtractor.extractTableSchema(clazz);
        } else {
            throw new IllegalArgumentException("Can't construct YT schema for class " + clazz);
        }

    }

    public <T> YPath createTableWithSchema(YPath table, Class<T> clazz) {
        return createTableWithSchema(Option.empty(), table, clazz);
    }

    public <T> YPath createTableWithSchema(Option<GUID> transaction, YPath table, Class<T> clazz) {
        TableSchema schema = extractTableSchema(clazz);
        createTableWithSchema(transaction, table, schema);
        return table;
    }

    public YPath createTableWithSchema(YPath table, TableSchema schema) {
        return createTableWithSchema(Option.empty(), table, schema);
    }

    public YPath createTableWithSchema(Option<GUID> transaction, YPath table, TableSchema schema) {

        YTreeNode ysonSchema = schema.toYTree();

        LOG.info("Creating table {} with schema {}", table, ysonSchema);

        yt.cypress().remove(transaction.orElse((GUID) null), false, table);

        yt.cypress().create(
                transaction.toOptional(),
                false,
                table,
                CypressNodeType.TABLE,
                true,
                false,
                Map.of(
                        "schema", ysonSchema,
                        "optimize_for", YTree.stringNode("scan")
                ));

        return table;
    }

    public YPath createTableWithSchemaForDynamic(Option<GUID> transaction, YPath table, YTreeNode schema) {

        LOG.info("Creating table {} with schema {}, prepared to be dynamic", table, schema);

        yt.cypress().remove(transaction.orElse((GUID) null), false, table);

        yt.cypress().create(
                transaction.toOptional(),
                false,
                table,
                CypressNodeType.TABLE,
                true,
                false,
                Map.of(
                        "schema", schema,
                        "optimize_for", YTree.stringNode("lookup")
                ));

        return table;
    }

    public void remove(YPath path) {
        remove(Option.empty(), path);
    }

    public void remove(Option<GUID> transaction, YPath path) {
        LOG.info("Removing {} in tx {}", path, transaction);
        yt.cypress().remove(transaction.orElse((GUID) null), false, path, true, true);
    }

    public void move(Option<GUID> transaction, YPath source, YPath target) {
        LOG.info("Moving from {} to {} in tx {}", source, target, transaction);
        yt.cypress().move(transaction.orElse((GUID) null), false, source, target, true, true, false);
    }

    public void removeAll(CollectionF<YPath> paths) {
        for (YPath path : paths) {
            remove(path);
        }
    }

    public void copyDirContent(YPath fromDir, YPath toDir) {
        Transaction transaction = yt.transactions().startAndGet();
        LOG.info("Copy {} content to {} in transaction {}", fromDir, toDir, transaction.getId());

        ls(fromDir).forEach(from -> {
            YPath to = toDir.child(from.name());
            yt.cypress().copy(transaction.getId(), true, from, to, true, true, false);
        });

        transaction.commit();
    }

    public Option<LocalDate> getGenerationDate(YPath table) {
        YPath attributePath = table.attribute(GENERATION_DATE_ATTR);
        if (yt.cypress().exists(attributePath)) {
            YTreeNode dateAttr = yt.cypress().get(attributePath);
            String dateStr = dateAttr.stringValue();
            return Option.of(LocalDate.parse(dateStr));
        } else {
            return Option.empty();
        }

    }

    public void setGenerationDate(YPath table, LocalDate date) {
        yt.cypress().set(table.attribute(GENERATION_DATE_ATTR), date.toString());
    }

    public void setGenerationDate(Option<GUID> txId, YPath table, LocalDate date) {
        yt.cypress().set(txId.orElse((GUID) null), false, table.attribute(GENERATION_DATE_ATTR), date.toString());
    }

    public void setRunDate(YPath table, LocalDate date) {
        yt.cypress().set(table.attribute(RUN_DATE_ATTR), date.toString());
    }

    public void setRunDate(Option<GUID> txId, YPath table, LocalDate date) {
        yt.cypress().set(txId.orElse((GUID) null), false, table.attribute(RUN_DATE_ATTR), date.toString());
    }

    public void setAttribute(YPath table, String attributeName, String attributeValue) {
        yt.cypress().set(table.attribute(attributeName), attributeValue);
    }

    public void setAttribute(Option<GUID> txId, YPath table, String attributeName, String attributeValue) {
        yt.cypress().set(txId.orElse((GUID) null), false, table.attribute(attributeName), attributeValue);
    }
}
