package ru.yandex.crypta.graph2.dao.yt;

import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.dao.yt.ops.Await;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.transactions.Transaction;
import ru.yandex.yt.ytclient.tables.TableSchema;

public class YtTransactionContext {

    private static final Logger LOG = LoggerFactory.getLogger(YtTransactionContext.class);

    private final Dao dao;
    private final YPath tmpDir;
    private final Option<GUID> transactionId;

    public YtTransactionContext(Dao dao, YPath tmpDir, Transaction transactionId) {
        this.dao = dao;
        this.tmpDir = tmpDir;
        this.transactionId = Option.of(transactionId.getId());
    }

    public <T> TableWithTmp createWithSchema(YPath table, Class<T> clazz) {
        TableWithTmp tableWithTmp = new TableWithTmp(tmpDir, table);
        dao.ytCypress().createTableWithSchema(transactionId, tableWithTmp.getTmpTablePath(), clazz);
        // need to re-create original table to avoid schema sorting errors
        dao.ytCypress().createTableWithSchema(transactionId, tableWithTmp.getTablePath(), clazz);
        return tableWithTmp;
    }

    public TableWithTmp createWithSchema(YPath table, TableSchema schema) {
        TableWithTmp tableWithTmp = new TableWithTmp(tmpDir, table);
        dao.ytCypress().createTableWithSchema(transactionId, tableWithTmp.getTmpTablePath(), schema);
        // need to re-create original table to avoid schema sorting errors
        dao.ytCypress().createTableWithSchema(transactionId, tableWithTmp.getTablePath(), schema);
        return tableWithTmp;
    }

    public TableWithTmp createNoSchema(YPath table) {
        TableWithTmp tableWithTmp = new TableWithTmp(tmpDir, table);
        dao.yt().cypress().create(transactionId.toOptional(), false,
                tableWithTmp.getTmpTablePath(), CypressNodeType.TABLE,
                true, false, true, Map.of());
        return tableWithTmp;
    }

    public void finalizeTables(ListF<TableWithTmp> tables, boolean mergeChunks) {
        Await.executeAll(tables.map(t -> finalizeTableAsync(t, mergeChunks)));
    }

    public void finalizeTables(ListF<TableWithTmp> tables) {
        finalizeTables(tables, true);
    }

    private Runnable finalizeTableAsync(TableWithTmp tableWithTmp, boolean mergeChunks) {
        if (mergeChunks) {
            return dao.ytOps().mergeChunksAsync(
                    transactionId,
                    tableWithTmp.getTmpTablePath(),
                    tableWithTmp.getTmpTablePath()
            ).andThen(op -> {
                op.awaitAndThrowIfNotSuccess();
                LOG.info("Done merging chunks in {}", tableWithTmp.getTmpTablePath());
                dao.ytCypress().move(transactionId, tableWithTmp.getTmpTablePath(), tableWithTmp.getTablePath());
            });
        } else {
            return () -> dao.ytCypress().move(
                    transactionId,
                    tableWithTmp.getTmpTablePath(),
                    tableWithTmp.getTablePath()
            );
        }
    }

    public GUID getTransactionId() {
        return transactionId.get();
    }
}
