package ru.yandex.crypta.graph2.dao.yt;

import java.time.Duration;
import java.util.Optional;
import java.util.function.Consumer;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.tables.utils.YtTablesUtils;
import ru.yandex.inside.yt.kosher.impl.transactions.utils.YtTransactionsUtils;
import ru.yandex.inside.yt.kosher.transactions.Transaction;

public class YtTransactionsHelper {

    private static final Duration PING_TIMEOUT = Duration.ofSeconds(15);
    // allows transaction to miss 10 pings due to network issues and stay alive
    private static final Duration TRANSACTION_TIMEOUT = PING_TIMEOUT.multipliedBy(10);
    private final Yt yt;

    public YtTransactionsHelper(Yt yt) {
        this.yt = yt;
    }

    public <T> T withTransactionAndGet(Function<Transaction, T> callback) {
        // NOTICE: ping will be chosen as a half of timeout. We set it just as api workaround
        return YtTransactionsUtils.withTransaction(yt, TRANSACTION_TIMEOUT, Optional.of(PING_TIMEOUT), callback);
    }

    public <T> T withTransactionIdAndGet(Function<Option<GUID>, T> callback) {
        return withTransactionAndGet(transaction -> callback.apply(Option.of(transaction.getId())));
    }

    public void withTransactionId(Function1V<Option<GUID>> callback) {
        withTransaction(transaction -> callback.accept(Option.of(transaction.getId())));
    }

    public void withTransaction(Function1V<Transaction> callback) {
        withTransactionAndGet(callback.asFunctionReturnNull());
    }

    public void withTransactionAndTmpDir(Function2V<Transaction, YPath> callback) {
        withTransaction(tx ->
                YtTablesUtils.withNewTempDir(Optional.of(tx.getId()), yt, tmpDir -> {
                    callback.accept(tx, tmpDir);
                }));
    }

    public void withTransactionContext(Dao dao, Consumer<YtTransactionContext> callback) {
        // TODO: keep dao in helper
        withTransaction(tx ->
                YtTablesUtils.withNewTempDir(Optional.of(tx.getId()), dao.yt(), tmpDir -> {
                    YtTransactionContext context = new YtTransactionContext(dao, tmpDir, tx);
                    callback.accept(context);
                })
        );
    }

}
