package ru.yandex.crypta.graph2.dao.yt.local.fastyt.fs;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.serialization.YTreeTextSerializer;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.misc.ExceptionUtils;

import static java.nio.file.StandardOpenOption.CREATE;
import static java.nio.file.StandardOpenOption.TRUNCATE_EXISTING;

public class FileBasedLocalYtDataLayer implements LocalYtDataLayer<Path> {

    private static final Logger LOG = LoggerFactory.getLogger(FileBasedLocalYtDataLayer.class);

    private final Path baseDir;

    public FileBasedLocalYtDataLayer(Path baseDir) {
        this.baseDir = baseDir;
        LOG.info("Using local yt layer at " + baseDir.toAbsolutePath());
    }

    @Override
    public InputStream createInputStream(YPath path) {
        Path fsPath = convertToLocalNode(path);

        // to allow write source from file to itself
        Path tmpFsPath = fsPath.resolveSibling(fsPath.getFileName() + UUID.randomUUID().toString());

        try {
            Files.copy(fsPath, tmpFsPath, StandardCopyOption.REPLACE_EXISTING);

            InputStream inputStream = Files.newInputStream(tmpFsPath);
            return new DelegatingInputStream(inputStream) {
                @Override
                public void close() throws IOException {
                    super.close();
                    Files.deleteIfExists(tmpFsPath);
                }
            };
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @Override
    public OutputStream createOutputStream(YPath path) {
        Path fsPath = convertToLocalNode(path);

        try {
            Files.createDirectories(fsPath.getParent());
            return Files.newOutputStream(fsPath, CREATE, TRUNCATE_EXISTING);
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }

    }

    @Override
    public void setMetadata(YPath path, YTreeMapNode metadata) {
        Path fsPath = convertToLocalNode(path);
        Path metadataPath = fsPath.resolveSibling(fsPath.getFileName() + ".metadata");

        String metadataStr = YTreeTextSerializer.serialize(metadata);
        try (BufferedWriter writer = Files.newBufferedWriter(metadataPath)) {
            writer.write(metadataStr);
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @Override
    public Option<YTreeMapNode> getMetadata(YPath path) {
        Path fsPath = convertToLocalNode(path);
        Path metadataPath = fsPath.resolveSibling(fsPath.getFileName() + ".metadata");

        if (Files.exists(metadataPath)) {
            try (InputStream inputStream = Files.newInputStream(metadataPath)) {
                return Option.of(YTreeTextSerializer.deserialize(inputStream).mapNode());
            } catch (IOException e) {
                throw ExceptionUtils.translate(e);
            }
        } else {
            return Option.empty();
        }

    }

    @Override
    public Path convertToLocalNode(YPath ypath) {
        String ypathStr = ypath.justPath().toString();
        String subPath = ypathStr.substring(2);
        return baseDir.resolve(subPath);

    }
}
