package ru.yandex.crypta.graph2.dao.yt.local.fastyt.ops;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Stream;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.dao.yt.local.StatisticsSlf4jLoggingImpl;
import ru.yandex.crypta.graph2.dao.yt.local.fastyt.client.YtOperationsFakeImpl;
import ru.yandex.crypta.graph2.dao.yt.local.fastyt.fs.LocalYtDataLayer;
import ru.yandex.crypta.graph2.dao.yt.local.fastyt.recs.YsonRecsLayer;
import ru.yandex.crypta.graph2.dao.yt.local.fastyt.recs.YsonToEntityTransformer;
import ru.yandex.crypta.graph2.dao.yt.local.fastyt.recs.YsonToOneOfProtoTransformer;
import ru.yandex.crypta.graph2.dao.yt.local.fastyt.recs.YsonToSingleProtoTransformer;
import ru.yandex.crypta.graph2.dao.yt.local.fastyt.testdata.SortInfoHelper;
import ru.yandex.crypta.graph2.dao.yt.ops.JoinReduceOperation;
import ru.yandex.crypta.graph2.dao.yt.ops.MapOperation;
import ru.yandex.crypta.graph2.dao.yt.ops.MapReduceOperation;
import ru.yandex.crypta.graph2.dao.yt.ops.MergeOperation;
import ru.yandex.crypta.graph2.dao.yt.ops.ReduceOperation;
import ru.yandex.crypta.graph2.dao.yt.ops.SortOperation;
import ru.yandex.crypta.graph2.dao.yt.ops.YtOps;
import ru.yandex.crypta.graph2.dao.yt.proto.NativeProtobufOneOfMessageEntryType;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.operations.utils.ReducerWithKey;
import ru.yandex.inside.yt.kosher.operations.Operation;
import ru.yandex.inside.yt.kosher.operations.OperationContext;
import ru.yandex.inside.yt.kosher.operations.Yield;
import ru.yandex.inside.yt.kosher.operations.map.Mapper;
import ru.yandex.inside.yt.kosher.operations.reduce.Reducer;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;
import ru.yandex.inside.yt.kosher.tables.types.NativeProtobufEntryType;
import ru.yandex.inside.yt.kosher.tables.types.YTreeObjectEntryType;
import ru.yandex.inside.yt.kosher.tables.types.YsonTableEntryType;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.misc.ExceptionUtils;

import static ru.yandex.inside.yt.kosher.tables.YTableEntryTypes.YSON;

/**
 * Emulated map-reduce operations using local file-system
 * Keeps data in yson in underlying dataLayer and transform to yson and back for every rec type.
 * Keeps
 */
public class YtOpsLocalImpl implements YtOps {

    // TODO: move it to YtOperations

    private final LocalYtDataLayer dataLayer;
    private final SortInfoHelper sortInfoHelper;
    private final YsonRecsLayer recsLayer;


    public YtOpsLocalImpl(LocalYtDataLayer dataLayer) {
        this.dataLayer = dataLayer;
        this.sortInfoHelper = new SortInfoHelper(dataLayer);
        this.recsLayer = new YsonRecsLayer(dataLayer);
    }

    private Stream fromYson(Stream<YTreeMapNode> recs, YTableEntryType entryType) {
        if (entryType instanceof NativeProtobufOneOfMessageEntryType) {
            NativeProtobufOneOfMessageEntryType protoEntryType = (NativeProtobufOneOfMessageEntryType) entryType;
            YsonToOneOfProtoTransformer transformer = new YsonToOneOfProtoTransformer(
                    protoEntryType.getBuilder(),
                    protoEntryType.getOneofDescriptor()
            );
            return recs.map(transformer::fromYson);
        } else if (entryType instanceof NativeProtobufEntryType) {
            NativeProtobufEntryType protoEntryType = (NativeProtobufEntryType) entryType;
            YsonToSingleProtoTransformer transformer = new YsonToSingleProtoTransformer(
                    protoEntryType.getBuilder()
            );
            return recs.map(transformer::fromYson);
        }
        if (entryType instanceof YsonTableEntryType) {
            return recs;
        } else if (entryType instanceof YTreeObjectEntryType) {
            YTreeObjectEntryType entityEntryType = (YTreeObjectEntryType) entryType;
            YsonToEntityTransformer transformer = new YsonToEntityTransformer(entityEntryType.getSerializer());
            return recs.map(r -> transformer.fromYson(r));
        } else {
            throw new IllegalStateException("entryType is not supported: " + entryType);
        }
    }

    private <T> Yield<T> getYield(OutputStream[] outputs, YTableEntryType<T> entryType) {
        Yield<YTreeMapNode> ysonYield = YSON.yield(outputs);
        if (entryType instanceof NativeProtobufOneOfMessageEntryType) {
            NativeProtobufOneOfMessageEntryType protoEntryType = (NativeProtobufOneOfMessageEntryType) entryType;
            YsonToOneOfProtoTransformer transformer = new YsonToOneOfProtoTransformer(
                    protoEntryType.getBuilder(),
                    protoEntryType.getOneofDescriptor()
            );
            return (Yield<T>) transformer.yield(ysonYield);

        } else if (entryType instanceof NativeProtobufEntryType) {
            NativeProtobufEntryType protoEntryType = (NativeProtobufEntryType) entryType;
            YsonToSingleProtoTransformer transformer = new YsonToSingleProtoTransformer(
                    protoEntryType.getBuilder()
            );
            return (Yield<T>) transformer.yield(ysonYield);
        } else if (entryType instanceof YsonTableEntryType) {
            return (Yield<T>) ysonYield;
        } else if (entryType instanceof YTreeObjectEntryType) {
            YTreeObjectEntryType entityEntryType = (YTreeObjectEntryType) entryType;
            YsonToEntityTransformer transformer = new YsonToEntityTransformer(entityEntryType.getSerializer());
            return (Yield<T>) transformer.yield(ysonYield);
        } else {
            throw new IllegalStateException("entryType is not supported: " + entryType);
        }
    }


    @Override
    public ReduceOperation reduceOperation(ListF<YPath> inputTables, ListF<YPath> outputTables,
                                           ListF<String> reduceBy, Reducer reducer) {
        return new ReduceOperation(null, inputTables, outputTables, reduceBy, reducer) {
            @Override
            public Operation runAndGetOp(Option<GUID> transactionId) {

                for (YPath inputTable : inputTables) {
                    List<String> sortedBy = sortInfoHelper.getSortedBy(inputTable);
                    if (sortedBy.isEmpty() || !sortedBy.equals(reduceBy)) {
                        throw new IllegalStateException("Table " + inputTable +
                                " should be sorted by the reduce key " + reduceBy +
                                " but sorted by " + sortedBy);
                    }
                }

                if (!(reducer instanceof ReducerWithKey)) {
                    throw new IllegalArgumentException("Only ReducerWithKey is supported");
                }

                try (Stream<YTreeMapNode> ysonRecs = recsLayer.readReduce(inputTables, reduceBy)) {
                    Stream inputRecs = fromYson(ysonRecs, reducer.inputType());

                    OutputStream[] outputs = outputTables
                            .map(dataLayer::createOutputStream)
                            .toArray(OutputStream.class);

                    try (Yield yield = getYield(outputs, reducer.outputType())) {
                        reducer.reduce(inputRecs.iterator(), yield, new StatisticsSlf4jLoggingImpl(),
                                new OperationContext());
                    } catch (IOException e) {
                        throw ExceptionUtils.translate(e);
                    }
                } catch (Exception e) {
                    throw ExceptionUtils.translate(e);
                }


                return new YtOperationsFakeImpl.OperationStub();
            }
        };

    }

    @Override
    public JoinReduceOperation joinReduceOperation(ListF<YPath> inputTables, ListF<YPath> outputTables,
                                                   ListF<String> joinBy, Reducer reducer) {

        // substitute with just reduce as it has even stricter semantics
        ReduceOperation reduceOperation = reduceOperation(inputTables, outputTables, joinBy, reducer);

        return new JoinReduceOperation(null, inputTables, outputTables, joinBy, reducer) {
            @Override
            public Operation runAndGetOp(Option<GUID> transactionId) {
                return reduceOperation.runAndGetOp(transactionId);
            }
        };
    }


    @Override
    public MapOperation mapOperation(ListF<YPath> inputTables, ListF<YPath> outputTables, Mapper mapper) {
        return new MapOperation(null, inputTables, outputTables, mapper) {
            @Override
            public Operation runAndGetOp(Option<GUID> transactionId) {

                OutputStream[] outputs = outputTables.map(dataLayer::createOutputStream).toArray(OutputStream.class);

                try (Stream<YTreeMapNode> ysonRecs = recsLayer.readMap(inputTables)) {
                    Stream inputRecs = fromYson(ysonRecs, mapper.inputType());
                    try (Yield yield = getYield(outputs, mapper.outputType())) {
                        inputRecs.forEach(r -> mapper.map(r, yield, new StatisticsSlf4jLoggingImpl()));
                    } catch (Exception e) {
                        throw ExceptionUtils.translate(e);
                    }
                } catch (Exception e) {
                    throw ExceptionUtils.translate(e);
                }


                return new YtOperationsFakeImpl.OperationStub();
            }
        };
    }

    @Override
    public MapReduceOperation mapReduceOperation(ListF<YPath> inputTables, ListF<YPath> outputTables, Mapper mapper,
                                                 ListF<String> reduceBy, Reducer reducer) {
        throw new UnsupportedOperationException();
    }

    private <T> void writeToOutput(Stream<T> outputData, YPath outputTable, YTableEntryType<T> entryType) {

        OutputStream outputStream = dataLayer.createOutputStream(outputTable);

        try (Yield<T> yield = getYield(new OutputStream[]{outputStream}, entryType)) {
            outputData.forEach(yield::yield);
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private String extractKey(YTreeMapNode rec, ListF<String> sortedBy) {
        return sortedBy.map(rec::getString).mkString("");
    }

    @Override
    public SortOperation sortOperation(ListF<YPath> source, YPath target, ListF<String> keys) {
        final Comparator<YTreeMapNode> recsComparator = Comparator.comparing(rec -> extractKey(rec, keys));

        return new SortOperation(null, source, target, keys) {
            @Override
            public Operation runAndGetOp(Option<GUID> transactionId) {


                try (Stream<YTreeMapNode> recsStream = recsLayer.readMap(source)) {
                    writeToOutput(recsStream.sorted(recsComparator), target, YSON);
                    sortInfoHelper.setSortedBy(target, keys);
                } catch (Exception e) {
                    throw ExceptionUtils.translate(e);
                }

                return new YtOperationsFakeImpl.OperationStub();
            }
        };
    }

    @Override
    public MergeOperation mergeChunksOperation(ListF<YPath> source, YPath target) {
        return new MergeOperation(null, source, target) {
            @Override
            public Operation runAndGetOp(Option<GUID> transactionId) {

                try (Stream<YTreeMapNode> sorted = recsLayer.readMap(source)) {
                    writeToOutput(sorted, target, YSON);
                } catch (Exception e) {
                    throw ExceptionUtils.translate(e);
                }

                return new YtOperationsFakeImpl.OperationStub();

            }
        };
    }

}
