package ru.yandex.crypta.graph2.dao.yt.local.fastyt.recs;

import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.PriorityQueue;
import java.util.Queue;

import com.google.common.collect.Iterators;
import com.google.common.collect.PeekingIterator;
import com.google.common.collect.UnmodifiableIterator;

/**
 * Resulting iterator produces merged output in the following order:
 * 1. by key ascending
 * 2. by iterator index withing single key
 * <p>
 * This order corresponds to YT reducer order guaranties.
 * It assumes that all inner iterators are sorted by key.
 * <p>
 * TODO: there is a simpler way to do this just by reading sequential data without sorting heap
 *
 * @param <R> element type
 */
public class MergingIteratorByKeyAndTableIndex<R> extends UnmodifiableIterator<RecWithKeyAndTableIndex<R>> {

    private final Queue<PeekingIterator<RecWithKeyAndTableIndex<R>>> queue;

    public MergingIteratorByKeyAndTableIndex(List<Iterator<RecWithKeyAndTableIndex<R>>> iterators) {

        // sort recs by key and table index
        Comparator<RecWithKeyAndTableIndex<R>> byKeyAndTableIndexComparator = Comparator
                .<RecWithKeyAndTableIndex<R>, String>comparing(RecWithKeyAndTableIndex::getKey)
                .thenComparingInt(RecWithKeyAndTableIndex::getTableIndex);

        // sort iterators by the top (peek) element
        Comparator<PeekingIterator<RecWithKeyAndTableIndex<R>>> heapComparator = Comparator.comparing(
                PeekingIterator::peek, byKeyAndTableIndexComparator
        );

        queue = new PriorityQueue<>(2, heapComparator);

        for (Iterator<RecWithKeyAndTableIndex<R>> tableIterator : iterators) {
            if (tableIterator.hasNext()) {
                queue.add(Iterators.peekingIterator(tableIterator));
            }
        }

    }

    @Override
    public boolean hasNext() {
        return !queue.isEmpty();
    }

    @Override
    public RecWithKeyAndTableIndex<R> next() {
        PeekingIterator<RecWithKeyAndTableIndex<R>> nextIter = queue.remove();
        RecWithKeyAndTableIndex<R> next = nextIter.next();
        if (nextIter.hasNext()) {
            queue.add(nextIter);
        }
        return next;
    }


}

