package ru.yandex.crypta.graph2.dao.yt.local.fastyt.recs;

import java.io.IOException;

import com.google.protobuf.Descriptors;
import com.google.protobuf.Message;

import ru.yandex.inside.yt.kosher.impl.ytree.YTreeProtoUtils;
import ru.yandex.inside.yt.kosher.operations.Yield;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

import static ru.yandex.crypta.graph2.dao.yt.bendable.YsonMultiEntitySupport.TABLE_INDEX_COLUMN;

/**
 * Converts rec between yson and one-of protobuf rec,
 * in which several types of recs are represented with single proto message with one-of field
 */
public class YsonToOneOfProtoTransformer implements YsonRecsTransformer<Message> {

    private final Message.Builder baseBuilder;
    private final Descriptors.OneofDescriptor oneofDescriptor;

    private int getTableIndex(YTreeMapNode rec) {
        return rec.getAttributeOrThrow(TABLE_INDEX_COLUMN).intValue();
    }

    public YsonToOneOfProtoTransformer(Message.Builder baseBuilder,
                                       Descriptors.OneofDescriptor oneofDescriptor) {
        this.baseBuilder = baseBuilder;
        this.oneofDescriptor = oneofDescriptor;
    }

    @Override
    public Message fromYson(YTreeMapNode rec) {
        int tableIndex = getTableIndex(rec);
        Descriptors.FieldDescriptor descriptor = oneofDescriptor.getField(tableIndex);
        Message.Builder builder = baseBuilder.getFieldBuilder(descriptor);

        builder.clear();
        Message oneOfRec = YTreeProtoUtils.unmarshal(rec, builder);

        baseBuilder.clear();
        return baseBuilder.setField(descriptor, oneOfRec).build();
    }

    @Override
    public YTreeMapNode toYson(Message rec, int tableIndex) {
        Descriptors.FieldDescriptor descriptor = oneofDescriptor.getField(tableIndex);
        Message message = (Message) rec.getField(descriptor);
        return (YTreeMapNode) YTreeProtoUtils.marshal(message);
    }

    @Override
    public Yield<Message> yield(Yield<YTreeMapNode> yield) {
        return new Yield<>() {
            @Override
            public void yield(int index, Message value) {
                yield.yield(index, toYson(value, index));
            }

            @Override
            public void yield(Message value) {
                Descriptors.FieldDescriptor field = value.getOneofFieldDescriptor(oneofDescriptor);
                int tableIndex = field.getIndex();
                this.yield(tableIndex, value);
            }

            @Override
            public void close() throws IOException {
                yield.close();
            }
        };
    }

}
