package ru.yandex.crypta.graph2.dao.yt.ops;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.DataSize;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.common.JavaOptions;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.operations.Operation;
import ru.yandex.inside.yt.kosher.operations.map.Mapper;
import ru.yandex.inside.yt.kosher.operations.specs.JobIo;
import ru.yandex.inside.yt.kosher.operations.specs.MapSpec;
import ru.yandex.inside.yt.kosher.operations.specs.MapperSpec;
import ru.yandex.inside.yt.kosher.tables.TableWriterOptions;

public class MapOperation extends RichYtOperation {

    public static final DataSize DEFENSIVE_MAPPER_MEMORY_LIMIT = MapperSpec.DEFAULT_MEMORY_LIMIT
            // additional gap above the default Xmx to avoid OOM
            .plus(DataSize.fromMegaBytes(100));
    // assumes 20MB/s job takes ~3min
    public static final DataSize DEFAULT_MAP_DATA_SIZE_PER_JOB = DataSize.fromGigaBytes(3);

    private static final Logger LOG = LoggerFactory.getLogger(MapOperation.class);

    private Yt yt;
    private MapperSpec.Builder mapperSpecBuilder;
    private MapSpec.Builder mapSpecBuilder;

    public MapOperation(Yt yt, ListF<YPath> inputTables, ListF<YPath> outputTables, Mapper mapper) {
        this.yt = yt;

        mapperSpecBuilder = MapperSpec.builder()
                .setMapper(mapper)
                .setJobTimeLimit(Option.of(DEFENSIVE_JOB_TIME_LIMIT.toMillis()).getOrNull())
                .setMemoryLimit(DEFENSIVE_MAPPER_MEMORY_LIMIT)
                .setJavaOptions(JavaOptions.empty()
                        .withXmx(MapperSpec.DEFAULT_MEMORY_LIMIT)
                        .withOption(LOCAL_LIBRARY_PATH_OPTION) // allows operation to lookup uploaded .so files locally
                );

        mapSpecBuilder = MapSpec.builder()
                .setInputTables(inputTables)
                .setOutputTables(outputTables)
                .setDataSizePerJob(DEFAULT_MAP_DATA_SIZE_PER_JOB)
                .setJobIo(new JobIo(new TableWriterOptions(DataSize.fromMegaBytes(32))));

        String scriptName = CurrentScriptName.get();
        setScriptName(mapSpecBuilder, scriptName);
    }

    public MapperSpec.Builder getMapperSpecBuilder() {
        return mapperSpecBuilder;
    }

    public MapSpec.Builder getMapSpecBuilder() {
        return mapSpecBuilder;
    }

    @Override
    public Operation runAndGetOp(Option<GUID> transactionId) {
        MapperSpec mapperSpec = mapperSpecBuilder.build();
        String description = mapperSpec.getMapperOrReducerTitle();
        LOG.info("Starting map " + description);

        mapSpecBuilder.setMapperSpec(mapperSpec);
        return yt.operations().mapAndGetOp(transactionId.toOptional(), true, mapSpecBuilder.build());
    }

}
