package ru.yandex.crypta.graph2.dao.yt.ops;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.DataSize;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.common.JavaOptions;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.operations.Operation;
import ru.yandex.inside.yt.kosher.operations.map.Mapper;
import ru.yandex.inside.yt.kosher.operations.reduce.Reducer;
import ru.yandex.inside.yt.kosher.operations.specs.MapReduceSpec;
import ru.yandex.inside.yt.kosher.operations.specs.MapperSpec;
import ru.yandex.inside.yt.kosher.operations.specs.ReducerSpec;

public class MapReduceOperation extends RichYtOperation {

    private static final Logger LOG = LoggerFactory.getLogger(MapReduceOperation.class);

    private Yt yt;

    private MapperSpec.Builder mapperSpecBuilder;
    private ReducerSpec.Builder reducerSpecBuilder;
    private MapReduceSpec.Builder mapReduceSpecBuilder;

    public MapReduceOperation(Yt yt, ListF<YPath> inputTables, ListF<YPath> outputTables,
                              Mapper mapper, ListF<String> reduceBy, Reducer reducer) {
        this.yt = yt;

        reducerSpecBuilder = ReducerSpec.builder()
                .setReducer(reducer)
                .setJobTimeLimit(Option.of(DEFENSIVE_JOB_TIME_LIMIT.toMillis()).getOrNull())
                .setMemoryLimit(DataSize.fromBytes(DEFENSIVE_REDUCER_MEMORY_LIMIT.toBytes()))
                .setJavaOptions(JavaOptions.empty()
                        .withXmx(DataSize.fromBytes(DEFAULT_REDUCER_MEMORY_LIMIT.toBytes()))
                        .withOption(LOCAL_LIBRARY_PATH_OPTION) // allows operation to lookup uploaded .so files locally
                );

        if (mapper != null) {
            mapperSpecBuilder = MapperSpec.builder()
                    .setMapper(mapper)
                    .setMemoryLimit(DataSize.fromBytes(DEFAULT_REDUCER_MEMORY_LIMIT.toBytes()))
                    .setJobTimeLimit(Option.of(DEFENSIVE_JOB_TIME_LIMIT.toMillis()).getOrNull());
        }

        mapReduceSpecBuilder = MapReduceSpec.builder()
                .setInputTables(inputTables)
                .setOutputTables(outputTables)
                .setReduceBy(reduceBy);

        String scriptName = CurrentScriptName.get();
        if (scriptName != null) {
            setScriptName(mapReduceSpecBuilder, scriptName);
        }

    }

    public MapperSpec.Builder getMapperSpecBuilder() {
        return mapperSpecBuilder;
    }

    public ReducerSpec.Builder getReducerSpecBuilder() {
        return reducerSpecBuilder;
    }

    public MapReduceSpec.Builder getMapReduceSpecBuilder() {
        return mapReduceSpecBuilder;
    }

    @Override
    public Operation runAndGetOp(Option<GUID> transactionId) {
        MapperSpec mapperSpec = null;
        if (mapperSpecBuilder != null) {
            mapperSpec = mapperSpecBuilder.build();
        }
        ReducerSpec reducerSpec = reducerSpecBuilder.build();

        String description = "MR(" +
                (mapperSpec == null ? "EMPTY_MAPPER" : mapperSpec.getMapperOrReducerTitle()) +
                " & " +
                reducerSpec.getMapperOrReducerTitle() +
                ")";
        LOG.info("Starting map reduce " + description);

        mapReduceSpecBuilder
                .setMapperSpec(mapperSpec)
                .setReducerSpec(reducerSpec);

        return yt.operations().mapReduceAndGetOp(transactionId.toOptional(), true, mapReduceSpecBuilder.build());
    }


}
