package ru.yandex.crypta.graph2.dao.yt.schema.extractor;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.yt.kosher.impl.ytree.object.FieldsBindingStrategy;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeFlattenField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeIgnoreField;
import ru.yandex.misc.enums.StringEnum;
import ru.yandex.misc.reflection.ClassX;
import ru.yandex.misc.reflection.FieldX;
import ru.yandex.misc.reflection.TypeX;
import ru.yandex.yt.ytclient.tables.ColumnSchema;
import ru.yandex.yt.ytclient.tables.ColumnValueType;

public class YTreeYtSchemaExtractor extends AbstractYtSchemaExtractor {

    private static List<FieldX> getFields(ClassX<?> clazz, FieldsBindingStrategy strategy) {
        // copy paste of ru.yandex.inside.yt.kosher.impl.ytree.object.serializers.YTreeObjectSerializer.getFields()
        final List<FieldX> fields;
        if (strategy == FieldsBindingStrategy.ALL_FIELDS) {
            fields = clazz.getAllDeclaredInstanceFields();
        } else if (strategy == FieldsBindingStrategy.ANNOTATED_ONLY) {
            fields = clazz.getAllDeclaredInstanceFields()
                    .filter(f -> f.hasAnnotationInAnnotatedOrItsAnnotations(YTreeField.class));
        } else {
            throw new IllegalArgumentException("Bad strategy: " + strategy);
        }
        return fields
                .stream()
                .filter(m -> !m.isTransient())
                .filter(f -> !f.hasAnnotationInAnnotatedOrItsAnnotations(YTreeIgnoreField.class))
                .collect(Collectors.toList());
    }

    private ColumnValueType primitiveClassToYsonType(TypeX type) {
        Class clazz = type.erasure().wrapPrimitiveSafe().getClazz();
        if (clazz.equals(Integer.class) || clazz.equals(Long.class)) {
            return ColumnValueType.INT64;
        } else if (clazz.equals(Float.class) || clazz.equals(Double.class)) {
            return ColumnValueType.DOUBLE;
        } else if (clazz.equals(String.class) || StringEnum.class.isAssignableFrom(clazz)) {
            return ColumnValueType.STRING;
        } else if (clazz.equals(Boolean.class)) {
            return ColumnValueType.BOOLEAN;
        } else if (clazz.equals(Option.class)) {
            TypeX optionGenericType = type.getActualTypeArguments().first();
            return primitiveClassToYsonType(optionGenericType);
        } else {
            return ColumnValueType.ANY;
        }
    }

    private void fieldsToSchemaColumns(List<FieldX> fields, Consumer<ColumnSchema> typesConsumer) {
        // process via consumer to support recursion
        for (FieldX field : fields) {
            if (field.hasAnnotation(YTreeFlattenField.class)) {
                for (FieldX nestedField : field.getType().getAllDeclaredFields()) {
                    // recursively dangerous
                    fieldsToSchemaColumns(Cf.list(nestedField), typesConsumer);
                }
            } else if (field.hasAnnotation(YTreeField.class)) {
                YTreeField yTreeField = field.getAnnotation(YTreeField.class);
                String columnName = Objects.requireNonNull(yTreeField).key();
                if (YTreeField.AUTO_KEY.equals(columnName)) {
                    columnName = field.getName();
                }
                ColumnValueType columnType;
                if (field.hasAnnotation(CustomColumnType.class)) {
                    CustomColumnType customColumnType = field.getAnnotation(CustomColumnType.class);
                    columnType = Objects.requireNonNull(customColumnType).value();
                } else {
                    columnType = primitiveClassToYsonType(field.getGenericType());
                }

                typesConsumer.accept(new ColumnSchema(columnName, columnType));
            }
        }
    }

    @Override
    protected <T> List<ColumnSchema> extractColumns(Class<T> aClass) {
        ClassX<T> classX = ClassX.wrap(aClass);
        // TODO: extract real FieldsBindingStrategy
        List<FieldX> fields = getFields(classX, FieldsBindingStrategy.ANNOTATED_ONLY);

        List<ColumnSchema> result = new ArrayList<>();
        fieldsToSchemaColumns(fields, result::add);
        return result;
    }


}
