package ru.yandex.crypta.graph2.utils;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Map;
import java.util.Set;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;

public class MapCounter<K extends Comparable<K>, V> {

    private MapF<K, Set<V>> innerMap;

    private Comparator<Map.Entry<K, Set<V>>> comp = new Comparator<Map.Entry<K, Set<V>>>() {
        @Override
        public int compare(Map.Entry<K, Set<V>> o1, Map.Entry<K, Set<V>> o2) {
            // compare by the length of value set
            int valuesCompare = Integer.compare(o1.getValue().size(), o2.getValue().size());
            if (valuesCompare == 0) {
                // if the number of values are equal for each key, compare keys for consistency
                return o1.getKey().compareTo(o2.getKey());
            } else {
                return valuesCompare;
            }

        }

    };

    public MapCounter() {
        this.innerMap = Cf.hashMap();
    }

    public void add(K key, V value) {
        innerMap.putIfAbsent(key, Cf.hashSet());
        innerMap.getTs(key).add(value);
    }

    public void addAll(K key, Collection<? extends V> values) {
        innerMap.putIfAbsent(key, Cf.hashSet());
        innerMap.getTs(key).addAll(values);
    }

    public K getKeyWithMaxValues() {
        return Collections.max(innerMap.entrySet(), comp).getKey();
    }

    public MaxCountValue<K> getMaxCountValue() {
        Map.Entry<K, Set<V>> maxEntry = Collections.max(innerMap.entrySet(), comp);
        return new MaxCountValue<>(maxEntry.getKey(), maxEntry.getValue().size());
    }

    public K getKeyWithMinValues() {
        return Collections.min(innerMap.entrySet(), comp).getKey();
    }

    public int getKeyCount(K key) {
        return innerMap.getO(key).mapToOptionalInt(Set::size).orElse(0);
    }

    public MapF<K, Integer> getKeysCounts() {
        return Cf.wrap(innerMap).mapValues(Set::size);
    }

    public MapF<K, Set<V>> getKeysValues() {
        return innerMap;
    }

    public static class MaxCountValue<T> {

        private T value;
        private int count;

        private MaxCountValue(T value, int count) {
            this.value = value;
            this.count = count;
        }

        public T getValue() {
            return value;
        }

        public int getCount() {
            return count;
        }
    }
}
