package ru.yandex.crypta.graph2.workflow;

import org.joda.time.format.PeriodFormatter;
import org.joda.time.format.PeriodFormatterBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.dao.yt.ops.CurrentScriptName;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.time.Stopwatch;

public abstract class Task<InTables, OutTables, Params> {

    private static final Logger LOG = LoggerFactory.getLogger(Task.class);

    private static final PeriodFormatter PERIOD_FORMATTER = new PeriodFormatterBuilder()
            .printZeroAlways()
            .appendHours()
            .appendSeparator(":")
            .appendMinutes()
            .appendSeparator(":")
            .appendSeconds()
            .appendSeparator(":")
            .appendMillis3Digit().toFormatter();

    protected Dao dao;
    protected YPath workdir;
    protected Params params;
    protected String nameSuffix;

    protected Task(Dao dao, YPath workdir, Params params, String nameSuffix) {
        this.dao = dao;
        this.workdir = workdir;
        this.params = params;
        this.nameSuffix = nameSuffix;
    }

    protected Task(Dao dao, YPath workdir, Params params) {
        this(dao, workdir, params, "");
    }

    public void run(InTables inTables) {
        try {
            LOG.info("Running {}: {}", getName(), getDescription());
            Stopwatch timer = Stopwatch.createAndStart();

            CurrentScriptName.set(getName());
            runImpl(inTables);
            CurrentScriptName.reset();

            timer.stop();
            String durationStr = PERIOD_FORMATTER.print(timer.duration().toPeriod());
            LOG.info("Task {} is done in {}", getName(), durationStr);

            cleanup();
            LOG.info("Cleanup of {} is done", getName());

        } catch (Exception e) {
            throw ExceptionUtils.translate(e, "Task " + getName() + " is failed");
        }

    }

    protected abstract void runImpl(InTables inTables);

    protected void cleanup() {
    }

    public abstract OutTables getOutput();

    public abstract String getDescription();

    public String getName() {
        String name = getClass().getSimpleName();
        if (!this.nameSuffix.isEmpty()) {
            name = String.format("%s_%s", name, this.nameSuffix);
        }
        return name;
    }

    public YPath getWorkdir() {
        return workdir;
    }
}
