"""Task DirectByTypes.
   Create direct tables (type -> type)

Parameters:

    generate_date               date of data generation
    run_date                    current date
"""

import logging
import contextlib

import crypta.lib.python.bt.conf.conf as conf
import yt.wrapper as yt
from yt.wrapper import OperationsTracker

from crypta.lib.python.native_yt.proto import (
    create_schema,)

from crypta.graph.acl.matching_acl import get_acl
from crypta.graph.matching.direct.lib.native_operations import (
    TDirectByCryptaIdMapper,
    TDirectByTypesWithoutCryptaIdMapper,
    TCryptaidsMatchingMapper,
)
from crypta.graph.matching.direct.lib.native_operations import TDirectReducer
from crypta.graph.matching.direct.lib.utils import (
    create_sorted_tables,
    get_schema,
    get_schema_for_yuids,
    get_types_info_with_paths_with_cryptaid,
    get_types_info_with_paths_without_cryptaid,
    collect_all_public_edges_without_indexes,
)
from crypta.lib.python.bt.tasks import YtTask
from crypta.lib.python.bt.workflow import IndependentTask, Parameter
from crypta.lib.python.bt.workflow.targets.table import (
    Exists,
    HasAttribute,
    NotEmptyAndSorted,
)

from crypta.graph.matching.direct.proto.types_pb2 import (
    TDirectNeighbours,
)


logger = logging.getLogger(__name__)


class DirectByTypes(YtTask, IndependentTask):
    run_date = Parameter()
    generate_date = Parameter()

    @contextlib.contextmanager
    def run_context(self):
        with yt.TempTable() as self.direct_temp_path, yt.TempTable() as self.direct_cryptaid_id_temp_path:
            logger.info('Use temporary table for direct [%s]', str(self.direct_temp_path))
            logger.info('Use cryptaid temporary table for direct [%s]', str(self.direct_cryptaid_id_temp_path))
            with super(DirectByTypes, self).run_context() as ctx:
                yield ctx

    def run(self, **kwargs):
        self._make_temporary_tables()
        self.create_root()
        self.direct_with_cryptaid()
        self.direct_between_cryptaid()
        self.direct_without_cryptaid()

    def targets(self):
        _, paths_all_without_crypta_id = get_types_info_with_paths_without_cryptaid(self.base_path)
        _, paths_crypta_id = get_types_info_with_paths_with_cryptaid(self.base_path)
        targets = []
        for path in self.paths():
            targets.extend([
                HasAttribute(
                    self.yt,
                    path,
                    "generate_date",
                    self.generate_date
                ),
                NotEmptyAndSorted(self.yt, path)
            ])
        for path in self.public_paths():
            targets.extend([
                Exists(self.yt, path)
            ])
        targets.extend([
            HasAttribute(self.yt, path, 'generate_date', self.generate_date) for path in paths_all_without_crypta_id
        ])
        targets.append(HasAttribute(self.yt, self.base_path, 'generate_date', self.generate_date))
        targets.extend([
            HasAttribute(self.yt, path, 'generate_date', self.generate_date) for path in paths_crypta_id
        ])
        return iter(targets)

    @property
    def base_path(self):
        return conf.direct_config.paths.directbytypes.base_path

    @property
    def account(self):
        accounts = {
            'production': 'crypta-graph',  # TODO: this is temporary account, reconsider later
            'testing': 'crypta-graph-testing',
            'develop': 'crypta-team',
        }
        return accounts[conf.environment.mode]

    @property
    def owner(self):
        owners = {
            'production': 'robot-crypta',  # TODO: this is temporary account, reconsider later
            'testing': 'robot-crypta',
            'develop': conf.environment.username,
        }
        return owners[conf.environment.mode]

    @property
    def root_attributes(self):
        acl = get_acl().get_basic_acl()
        attributes = {
            'owner': self.owner,
            'generate_date': self.generate_date,
            'run_date': self.run_date,
            'account': self.account
        }
        if acl and conf.environment.mode != 'develop':
            attributes.update({'inherit_acl': False, 'acl': acl})
        return attributes

    def create_root(self):
        logger.info('Creating directory ' + self.base_path)
        self.yt.create(
            'map_node',
            self.base_path,
            force=True,
            recursive=True,
            attributes=self.root_attributes
        )

    def direct_without_cryptaid(self):
        logger.info('running direct by types without cryptaId')
        types_info_all_without_crypta_id, paths_all_without_crypta_id = \
            get_types_info_with_paths_without_cryptaid(self.base_path)

        create_sorted_tables(
            yt_client=self.yt,
            types_info=types_info_all_without_crypta_id,
            logger=logger,
            generate_date=self.generate_date,
            run_date=self.run_date,
            acl=get_acl(),
        )

        state_all_without_crypta_id = types_info_all_without_crypta_id.SerializeToString()
        direct_by_id_type_target_id_type_id = self.direct_temp_path
        logger.info('computing direct by all types without crypta_id from direct table')
        self.native_map(
            TDirectByTypesWithoutCryptaIdMapper,
            source=direct_by_id_type_target_id_type_id,
            destination=paths_all_without_crypta_id,
            state=state_all_without_crypta_id,
            spec={'ordered': True}
        )

    def direct_with_cryptaid(self):
        logger.info('running direct by types with cryptaId')
        types_info_crypta_id, paths_crypta_id = get_types_info_with_paths_with_cryptaid(self.base_path)
        create_sorted_tables(
            yt_client=self.yt,
            types_info=types_info_crypta_id,
            logger=logger,
            generate_date=self.generate_date,
            run_date=self.run_date,
            acl=get_acl(),
        )

        state_crypta_id = types_info_crypta_id.SerializeToString()

        logger.info('computing direct by crypta_id')

        self.native_map(
            TDirectByCryptaIdMapper,
            source=self.direct_cryptaid_id_temp_path,
            destination=paths_crypta_id,
            state=state_crypta_id,
            spec={'ordered': True},
        )

    def paths(self):
        base_path = conf.direct_config.paths.directbytypes.base_path
        crypta_id_to_crypta_id1 = yt.ypath_join(base_path, 'crypta_id', 'crypta_id1')
        crypta_id1_to_crypta_id = yt.ypath_join(base_path, 'crypta_id1', 'crypta_id')
        return [crypta_id_to_crypta_id1, crypta_id1_to_crypta_id]

    def public_paths(self):
        public_path = conf.direct_config.paths.directbytypes.public_path
        public_crypta_id_to_crypta_id1 = yt.ypath_join(public_path, 'crypta_id', 'crypta_id1')
        public_crypta_id1_to_crypta_id = yt.ypath_join(public_path, 'crypta_id1', 'crypta_id')
        return [public_crypta_id_to_crypta_id1, public_crypta_id1_to_crypta_id]

    def attributes(self, path):
        attributes = {
            'schema': get_schema(),
            'optimize_for': 'scan',
            'generate_date': self.generate_date,
            'run_date': self.run_date,
        }
        if conf.environment.mode != 'develop':
            attributes.update({
                'acl': get_acl().get_pair_acl(
                    yt.ypath_split(yt.ypath_split(path)[0])[1],
                    yt.ypath_split(path)[1]
                ),
                'inherit_acl': False,
            })
        return attributes

    def direct_between_cryptaid(self):
        for path in self.paths():
            logger.info('Creating table ' + path)
            self.yt.create(
                'map_node',
                yt.ypath_dirname(path),
                ignore_existing=True,
                recursive=True,
            )
            self.yt.create(
                'table',
                path,
                attributes=self.attributes(path),
                force=True,
                recursive=True,
            )

        version_edges = conf.direct_config.paths.directbytypes.version_edges
        self.native_map(
            TCryptaidsMatchingMapper,
            source=version_edges,
            destination=self.paths(),
        )

        for path, public_path in zip(self.paths(),
                                     self.public_paths()):
            self.yt.run_sort(
                path,
                sort_by=['id', 'id_type'],
                spec={'annotations': {'script_name': 'DirectByTypes'}}
            )
            self.yt.create(
                'map_node',
                yt.ypath_dirname(public_path),
                recursive=True,
                ignore_existing=True,
            )
            logger.info('Creating link to ' + path + '. Link path ' + public_path)
            self.yt.link(path, public_path, force=True)

    @property
    def sort_by_id_type_and_id(self):
        return ['id_type', 'id']

    @property
    def sort_by_id_type_target_id_type_id(self):
        return ['id_type', 'target_id_type', 'id']

    @property
    def attributes_common(self):
        return {
            "schema": get_schema(),
            "optimize_for": 'scan',
        }

    @property
    def attributes_yandexuid(self):
        return {
            "schema": get_schema_for_yuids(),
            "optimize_for": 'scan',
        }

    def _make_temporary_tables(self):
        edges_by_crypta_id = conf.direct_config.paths.direct.edges_by_crypta_id
        direct_yandexuid_by_id_type_and_id = conf.direct_config.paths.direct.direct_yandexuid_by_id_type_and_id
        direct_by_id_type_target_id_type_id = self.direct_temp_path
        direct_cryptaid_id = self.direct_cryptaid_id_temp_path

        logger.info(
            'Computing direct graph from %s to %s and %s',
            edges_by_crypta_id,
            direct_by_id_type_target_id_type_id,
            direct_yandexuid_by_id_type_and_id
        )

        attributes = {
            "schema": create_schema(TDirectNeighbours),
            "optimize_for": 'scan',
        }
        self.yt.create(
            'table',
            direct_by_id_type_target_id_type_id,
            attributes=attributes,
            recursive=True,
            force=True
        )

        self.yt.create(
            'table',
            direct_yandexuid_by_id_type_and_id,
            attributes=self.attributes_yandexuid,
            force=True
        )

        self.yt.create(
            'table',
            direct_cryptaid_id,
            attributes=self.attributes_common,
            force=True
        )

        self.yt.set_attribute(direct_yandexuid_by_id_type_and_id, "generate_date", self.generate_date)
        self.yt.set_attribute(direct_yandexuid_by_id_type_and_id, "run_date", self.run_date)

        types_info_all_pulic_edges = collect_all_public_edges_without_indexes()

        self.native_reduce(
            TDirectReducer,
            source=edges_by_crypta_id,
            destination=[
                direct_by_id_type_target_id_type_id,
                direct_yandexuid_by_id_type_and_id,
                direct_cryptaid_id,
            ],
            reduce_by="cryptaId",
            state=types_info_all_pulic_edges.SerializeToString()
        )

        op_tracker = OperationsTracker()

        op_tracker.add(
            self.sort(
                source=direct_by_id_type_target_id_type_id,
                destination=direct_by_id_type_target_id_type_id,
                sort_by=['IDTypeNumber', 'ID'],
                sync=False
            )
        )

        op_tracker.add(
            self.sort(
                source=direct_cryptaid_id,
                destination=direct_cryptaid_id,
                sort_by=['id_type', 'target_id_type', 'id'],
                sync=False
            )
        )

        op_tracker.add(
            self.sort(
                source=direct_yandexuid_by_id_type_and_id,
                destination=direct_yandexuid_by_id_type_and_id,
                sort_by=self.sort_by_id_type_and_id,
                sync=False
            )
        )

        op_tracker.wait_all()
