#include "direct.h"


class TEdgeTypesInfo {
public:
    struct TEdgeTypeInfo {
        bool IsSoupyDirect = false;
        bool IsArtificial = false;
        bool IsFuzzy = false;
    };
    void Init() {
        TrustedTypesInfo.clear();
        const auto& edgeRecords = EdgeRecords();
        for (const auto& edgeRecord : edgeRecords ) {
            bool isArtificial = (edgeRecord.GetProps().GetEdgeStrength() == TEdgeProps::ARTIFICIAL);
            bool isSoupyIndevice = edgeRecord.GetUsage().GetSoupyIndevice();
            if (isSoupyIndevice || isArtificial) {
                TrustedTypesInfo[edgeRecord.GetType()] = {isSoupyIndevice, isArtificial};
            }
        }
    }

    TEdgeTypeInfo GetTypeInfo(const NIdType::TIdType& firstIDType, const NIdType::TIdType& secondIDType, const TEdge& edgeType) {
        TEdgeTypeInfo result{};
        TEdgeType edge;
        edge.SetId1Type(firstIDType.GetType());
        edge.SetId2Type(secondIDType.GetType());
        edge.SetLogSource(edgeType.LogSource.GetType());
        edge.SetSourceType(edgeType.SourceType.GetType());
        if (const auto it{TrustedTypesInfo.find(edge)}; it != TrustedTypesInfo.end()) {
            result = it->second;
        }
        edge.SetId1Type(secondIDType.GetType());
        edge.SetId2Type(firstIDType.GetType());

        if (const auto it{TrustedTypesInfo.find(edge)}; it != TrustedTypesInfo.end()) {
            result.IsArtificial = result.IsArtificial || (it->second).IsArtificial;
            result.IsSoupyDirect = result.IsSoupyDirect || (it->second).IsSoupyDirect;
        }
        if (edgeType.LogSource.GetType() == NLogSource::FUZZY2_INDEVICE) {
            result.IsFuzzy = true;
        }

        return result;
    }
private:
    TMap<TEdgeType, TEdgeTypeInfo> TrustedTypesInfo{};
};

TBoundaryTimestamps GetEmptyTimestamps() {
    return {
        .FirstTimestamp = 0,
        .LastTimestamp = 0
    };
}

struct TTargetMeta {
    TBoundaryTimestamps BoundaryTimestamps;
    bool IsArtificial = false;  //  flag for artificial edges without dates
    bool IsDirect = false;
    bool IsFuzzy = true;

    static TTargetMeta GetEmptyMeta(bool isArtificial = false) {
        return {
            .BoundaryTimestamps = GetEmptyTimestamps(),
            .IsArtificial = isArtificial
        };
    }
};

TBoundaryTimestamps JoinAsParallelTargets(const TBoundaryTimestamps& first, const TBoundaryTimestamps& second) {
    if (!first.LastTimestamp) {
        return second;
    }
    if (!second.LastTimestamp) {
        return first;
    }

    return {
        .FirstTimestamp = Min(first.FirstTimestamp, second.FirstTimestamp),
        .LastTimestamp = Max(first.LastTimestamp, second.LastTimestamp)
    };
}

TBoundaryTimestamps JoinAsSequencedTargets(const TBoundaryTimestamps& first, const TBoundaryTimestamps& second) {
    if (!first.LastTimestamp) {
        return first;
    }
    if (!second.LastTimestamp) {
        return second;
    }

    TBoundaryTimestamps result = {
        .FirstTimestamp = Max(first.FirstTimestamp, second.FirstTimestamp),
        .LastTimestamp = Min(first.LastTimestamp, second.LastTimestamp)
    };

    if (result.FirstTimestamp > result.LastTimestamp) {
        return GetEmptyTimestamps();
    }

    return result;
}

TTargetMeta JoinAsParallelTargets(const TTargetMeta& first, const TTargetMeta& second) {
    return {
        .BoundaryTimestamps = JoinAsParallelTargets(first.BoundaryTimestamps, second.BoundaryTimestamps),
        .IsArtificial = first.IsArtificial || second.IsArtificial,
        .IsDirect = first.IsDirect || second.IsDirect,
        .IsFuzzy = first.IsFuzzy && second.IsFuzzy,
    };
}

TTargetMeta JoinAsSequencedTargets(const TTargetMeta& first, const TTargetMeta& second) {
    if (first.IsArtificial && second.IsArtificial) {
        return JoinAsParallelTargets(first, second);
    }
    if (first.IsArtificial) {
        return second;
    }
    if (second.IsArtificial) {
        return first;
    }
    return {
        .BoundaryTimestamps = JoinAsSequencedTargets(first.BoundaryTimestamps, second.BoundaryTimestamps),
        .IsArtificial = first.IsArtificial && second.IsArtificial,
        .IsDirect = first.IsDirect && second.IsDirect,
        .IsFuzzy = first.IsFuzzy || second.IsFuzzy,
    };
}

TTargetMeta JoinAsSequencedTargets(const TTargetMeta& first, const TEdge& second, const TEdgeTypesInfo::TEdgeTypeInfo& typeInfo) {
    return JoinAsSequencedTargets(
        first,
        {
            .BoundaryTimestamps=second.BoundaryTimestamps,
            .IsArtificial=typeInfo.IsArtificial,
            .IsDirect=typeInfo.IsSoupyDirect,
            .IsFuzzy=typeInfo.IsFuzzy,
        }
    );
}


void UpdatePrivateYandexuids(THashMap<TString, bool>& privateYandexuids, const TVertex& vertex, const TNode& dates) {

    if (vertex.Type.GetType() == NIdType::YANDEXUID) {
        bool isNotPrivateYuid = !dates.IsNull() && !dates.AsList().empty() && !IsPrivate(dates.AsList(), vertex.Value);
        if (isNotPrivateYuid) {
            privateYandexuids[vertex.Value] = false;
        } else if (!privateYandexuids.contains(vertex.Value)) {
            privateYandexuids[vertex.Value] = true;
        }
    }
}

ui64 ConvertStringDateToTimestamp(const TString& date) {
    i64 timestamp;
    ParseISO8601DateTime(date.data(), timestamp);
    return static_cast<ui64>(timestamp);
}

TString ConvertTimestampToStringDate(ui64 timestamp) {
    tm timebuf;
    TInstant::Seconds(timestamp).GmTime(&timebuf);
    return Strftime("%Y-%m-%d", &timebuf);
}

TString TBoundaryTimestamps::GetDateBegin() const {
    if (FirstTimestamp) {
        return ConvertTimestampToStringDate(FirstTimestamp);
    }
    return "";
}

TString TBoundaryTimestamps::GetDateEnd() const {
    if (LastTimestamp) {
        return ConvertTimestampToStringDate(LastTimestamp);
    }
    return "";
}

bool TBoundaryTimestamps::IsEmpty() const {
    return (LastTimestamp == 0);
}

TBoundaryTimestamps GetBoundaryTimestamps(const TNode& rowDates) {

    auto boundaryTimestamps = GetEmptyTimestamps();

    if (!rowDates.IsNull() && rowDates.IsList() && !rowDates.AsList().empty()) {
        const auto& dates = rowDates.AsList();
        for (const auto& date : dates) {
            ui64 timestamp = ConvertStringDateToTimestamp(date.AsString());

            if (!boundaryTimestamps.FirstTimestamp || timestamp < boundaryTimestamps.FirstTimestamp) {
                boundaryTimestamps.FirstTimestamp = timestamp;
            }
            if (!boundaryTimestamps.LastTimestamp || timestamp > boundaryTimestamps.LastTimestamp) {
                boundaryTimestamps.LastTimestamp = timestamp;
            }
        }
    }

    return boundaryTimestamps;
}


bool IsEdgeWithPrivateYuid(const THashMap<TString, bool>& privateYandexuids, const TVertex& leftVertex, const TVertex& rightVertex) {
    bool isEdgePrivate = false;

    if (leftVertex.Type.GetType() == NIdType::YANDEXUID && privateYandexuids.at(leftVertex.Value)) {
        isEdgePrivate = true;
    }
    if (rightVertex.Type.GetType() == NIdType::YANDEXUID && privateYandexuids.at(rightVertex.Value)) {
        isEdgePrivate = true;
    }

    return isEdgePrivate;
}


class TCryptaProfile {
public:

    using TDirectTargets = TMap<TVertex, TTargetMeta>;
    struct TDirectVertexInfo {
        TDirectTargets Targets{};
        TTargetMeta Meta{};
    };

    TCryptaProfile(TTableReader<TNode>* input) {
        DirectTrustedEdgeTypesInfo.Init();
        for (bool first = true; input->IsValid(); input->Next(), first=false) {
            auto row = input->MoveRow();
            if (first) {
                CryptaID = FromString<ui64>(row["cryptaId"].AsString());
            }

            const auto& dates = row["dates"];
            const TVertex& vertexLeft = {
                IdType(row["id1Type"].ConvertTo<TString>()),
                row["id1"].ConvertTo<TString>()
            };
            const TVertex& vertexRight = {
                IdType(row["id2Type"].ConvertTo<TString>()),
                row["id2"].ConvertTo<TString>()
            };

            UpdatePrivateYandexuids(privateYandexuids, vertexLeft, dates);
            UpdatePrivateYandexuids(privateYandexuids, vertexRight, dates);


            const TEdge& source = {
                .SourceType = SourceType(row["sourceType"].ConvertTo<TString>()),
                .LogSource = LogSourceType(row["logSource"].ConvertTo<TString>()),
                .BoundaryTimestamps = GetBoundaryTimestamps(dates)
            };
            const TLinkedVertex& connectedToLeft = { vertexRight, source };
            const TLinkedVertex& connectedToRight = { vertexLeft, source };

            Graph[vertexLeft].push_back(connectedToLeft);
            Graph[vertexRight].push_back(connectedToRight);
        }
    }

    TDirectVertexInfo CollectDirectVertexInfo(const TVertex& vertex) {
        TDirectTargets directTargets;
        TQueue<TVertex> processedVertices;

        TTargetMeta meta = TTargetMeta::GetEmptyMeta(true);
        directTargets.emplace(vertex, meta);

        SpreadOverNeighbours(&directTargets, true, true);   // spread into direct trusted edges
        SpreadOverNeighbours(&directTargets, false, false); // add all adjacent vertices
        auto vertexMeta = CountMeta(directTargets);
        SpreadOverNeighbours(&directTargets, true, true);

        return {.Targets = directTargets, .Meta=vertexMeta};
    }

    TTargetMeta CountMeta(const TDirectTargets& directTargets) {
        TTargetMeta meta = TTargetMeta::GetEmptyMeta();
        for (const auto& item : directTargets) {
            meta = JoinAsParallelTargets(meta, item.second);
        }
        return meta;
    }

    template<class TWriter>
    void YieldDirectEdges(TWriter* output, const TTypesWithIndexes& typesInfo) {
         for (const auto& [vertex, connectedVertexSource] : Graph) {
            const auto& info = CollectDirectVertexInfo(vertex);
            YieldDirectEdges(vertex, info.Targets, output, typesInfo);
            YieldDirectEdgeWithCryptaID(vertex, info.Meta, output, typesInfo);
         }
    }

    template<class TWriter>
    void YieldDirectEdgeWithCryptaID(const TVertex& vertex, const TTargetMeta& meta, TWriter* output, const TTypesWithIndexes& typesInfo) {
        if (vertex.Type.GetType() == EIdType::YANDEXUID && privateYandexuids.at(vertex.Value)) {
            return;
        }
        TDirectEdge out;
        TString id = vertex.Value;
        TString idType = vertex.Type.GetName();
        TString stringCryptaID = ToString(CryptaID);
        if (!meta.BoundaryTimestamps.IsEmpty()) {
            out.SetDateBegin(meta.BoundaryTimestamps.GetDateBegin());
            out.SetDateEnd(meta.BoundaryTimestamps.GetDateEnd());
        }
        out.SetDirect(meta.IsDirect);
        out.SetFuzzy(meta.IsFuzzy);
        if (typesInfo.GetIndex(idType, "crypta_id") >= 0) {
            out.SetID(id);
            out.SetIDType(idType);
            out.SetTargetID(stringCryptaID);
            out.SetTargetIDType("crypta_id");

            output->AddRow(out, 2);
        }
        if (typesInfo.GetIndex("crypta_id", idType) >= 0) {
            out.SetID(stringCryptaID);
            out.SetIDType("crypta_id");
            out.SetTargetID(id);
            out.SetTargetIDType(idType);

            output->AddRow(out, 2);
        }
    }


    template<class TWriter>
    void YieldDirectEdgeWithYuid(const TVertex& vertex, const TString& yuid, const TTargetMeta& meta, TWriter* output) {
        TDirectEdge out;
        out.SetID(vertex.Value);
        out.SetIDType(vertex.Type.GetName());
        out.SetTargetID(yuid);
        if (!meta.BoundaryTimestamps.IsEmpty()) {
            out.SetDateBegin(meta.BoundaryTimestamps.GetDateBegin());
            out.SetDateEnd(meta.BoundaryTimestamps.GetDateEnd());
        }
        out.SetDirect(meta.IsDirect);
        out.SetFuzzy(meta.IsFuzzy);

        output->AddRow(out, DIRECT_YANDEXUID_BY_ID_TYPE_AND_ID_TABLE);
    }

    template<class TWriter>
    void YieldDirectEdges(const TVertex& vertex, const TDirectTargets& directTargets, TWriter* output, const TTypesWithIndexes& typesInfo) {
        TDirectNeighbours neighbours;
        neighbours.SetID(vertex.Value);
        neighbours.SetIDTypeNumber(static_cast<i64>(vertex.Type.GetType()));
        auto targets = neighbours.MutableTargets();

        for (const auto& item : directTargets) {
           const auto& connectedVertex = item.first;
           if (vertex == connectedVertex) {
                continue;
           }
           const auto& meta = item.second;

           if (!IsEdgeWithPrivateYuid(privateYandexuids, vertex, connectedVertex)) {
                if (connectedVertex.Type.GetType() == EIdType::YANDEXUID) {
                    YieldDirectEdgeWithYuid(vertex, connectedVertex.Value, meta, output);
                }
                if (typesInfo.Contains(vertex.Type.GetName(), connectedVertex.Type.GetName())) {
                    auto target = targets->AddValues();
                    target->SetID(connectedVertex.Value);
                    target->SetIDTypeNumber(static_cast<int>(connectedVertex.Type.GetType()));
                    if (!meta.BoundaryTimestamps.IsEmpty() && !meta.IsArtificial) {
                        target->SetFirstTimestamp(meta.BoundaryTimestamps.FirstTimestamp);
                        target->SetLastTimestamp(meta.BoundaryTimestamps.LastTimestamp);
                    }
                    target->SetDirect(meta.IsDirect);
                    target->SetFuzzy(meta.IsFuzzy);
                }
            }
        }
        Sort(targets->MutableValues()->begin(),
            targets->MutableValues()->end(),
            [](const TDirectNeighbours::TTarget& first, const TDirectNeighbours::TTarget& second) {
                return std::make_tuple(first.GetIDTypeNumber(), first.GetID()) < std::make_tuple(second.GetIDTypeNumber(), second.GetID());
            }
        );
        neighbours.SetDegree(neighbours.GetTargets().GetValues().size());
        if (!neighbours.GetDegree()) {
            return;
        }
        output->AddRow(neighbours, DIRECT_BY_ID_AND_ID_TYPE_TABLE);
    }

private:
    ui64 CryptaID = 0;
    TMap<TVertex, TVector<TLinkedVertex>> Graph{};
    THashMap<TString, bool> privateYandexuids{};
    TEdgeTypesInfo DirectTrustedEdgeTypesInfo{};

    TEdgeTypesInfo::TEdgeTypeInfo GetTypeInfo(const NIdType::TIdType& firstIDType, const NIdType::TIdType& secondIDType, const TEdge& edgeType) {
        return DirectTrustedEdgeTypesInfo.GetTypeInfo(firstIDType, secondIDType, edgeType);
    }

    void SpreadOverNeighbours(TDirectTargets* directTargets, bool directTrustedOnly, bool recursive) {
        TQueue<TVertex> processedVertices;
        for (const auto& item : (*directTargets)) {
            processedVertices.push(item.first);
        }

        while (!processedVertices.empty()) {
            auto curVertex = processedVertices.front();
            processedVertices.pop();
            auto currMetaIt = directTargets->find(curVertex);
            if (currMetaIt == directTargets->end()) {
                ythrow yexception() << "All vertices in queue must be contained in meta storage";
            }
            const auto& currMeta = currMetaIt->second;

            for (const auto& [connectedVertex, edgeType] : Graph.at(curVertex)) {
                const auto typeInfo = GetTypeInfo(curVertex.Type, connectedVertex.Type, edgeType);
                bool isTrusted = typeInfo.IsSoupyDirect || typeInfo.IsArtificial;
                TTargetMeta meta = JoinAsSequencedTargets(currMeta, edgeType, typeInfo);

                auto it = directTargets->find(connectedVertex);
                if (it != directTargets->end()) {
                    it->second = JoinAsParallelTargets(meta, it->second);
                    continue;
                }

                if (!directTrustedOnly || isTrusted) {
                    directTargets->emplace(connectedVertex, meta);
                    if (recursive) {
                        processedVertices.push(connectedVertex);
                    }
                }
            }
        }
    }
};


void TDirectReducer::Do(TTableReader<TNode>* input, TWriter* output) {
    TCryptaProfile profile(input);
    profile.YieldDirectEdges(output, TypesInfo);
}

void TDirectReducer::Start(TWriter*) {
    TypesInfo = TTypesWithIndexes(State);
}

void TDirectByTypesWithoutCryptaIdMapper::Do(TTableReader<TDirectNeighbours>* input, TTableWriter<TDirectEdge>* output) {
    auto *descriptor = NCrypta::NIdentifiersProto::NIdType::EIdType_descriptor();
    for (; input->IsValid(); input->Next()) {
        auto neighbours = input->MoveRow();
        TString idType = descriptor->FindValueByNumber(neighbours.GetIDTypeNumber())->options().GetExtension(NCrypta::NIdentifiersProto::Name);


        for (const auto& target : neighbours.GetTargets().GetValues()) {
            TString targetIDType = descriptor->FindValueByNumber(target.GetIDTypeNumber())->options().GetExtension(NCrypta::NIdentifiersProto::Name);

            auto tableIndex = TypesInfo.GetIndex(idType, targetIDType);
            if (tableIndex >= 0) {
                TDirectEdge out;
                out.SetID(neighbours.GetID());
                out.SetIDType(idType);
                out.SetTargetID(target.GetID());
                out.SetTargetIDType(targetIDType);
                if (target.GetLastTimestamp()) {
                    TBoundaryTimestamps boundaryTimestamps = {
                        .FirstTimestamp = target.GetFirstTimestamp(),
                        .LastTimestamp = target.GetLastTimestamp()
                    };
                    out.SetDateBegin(boundaryTimestamps.GetDateBegin());
                    out.SetDateEnd(boundaryTimestamps.GetDateEnd());
                }
                out.SetFuzzy(target.GetFuzzy());
                out.SetDirect(target.GetDirect());

                output->AddRow(out, tableIndex);
            }
        }
    }
}

void TDirectByTypesWithoutCryptaIdMapper::Start(TTableWriter<TDirectEdge>*) {
    TypesInfo = TTypesWithIndexes(State);
}

void TDirectByCryptaIdMapper::Do(TTableReader<TDirectEdge>* input, TTableWriter<TDirectEdge>* output) {

    for (; input->IsValid(); input->Next()) {
        auto row = input->GetRow();
        int tableIndex = TypesInfo.GetIndex(row.GetIDType(), row.GetTargetIDType());
        if (tableIndex >= 0) {
            output->AddRow(row, tableIndex);
        }
    }
}

void TDirectByCryptaIdMapper::Start(TTableWriter<TDirectEdge>*) {
    TypesInfo = TTypesWithIndexes(State);
}

void TCryptaidsMatchingMapper::Do(TTableReader<TSoupEdge>* input, TTableWriter<TDirectEdge>* output) {
    const int CRYPTAID_TO_CRYPTAID1 = 0;
    const int CRYPTAID1_TO_CRYPTAID = 1;
    for (; input->IsValid(); input->Next()) {
        auto row = input->MoveRow();
        TDirectEdge out;

        out.SetID(ToString(row.GetCryptaID()));
        out.SetIDType("crypta_id");
        out.SetTargetID(row.GetID2());
        out.SetTargetIDType("crypta_id1");

        output->AddRow(out, CRYPTAID_TO_CRYPTAID1);

        auto id = out.GetID();
        auto idType = out.GetIDType();
        out.SetID(out.GetTargetID());
        out.SetIDType(out.GetTargetIDType());
        out.SetTargetID(id);
        out.SetTargetIDType(idType);

        output->AddRow(out, CRYPTAID1_TO_CRYPTAID);

    }
}

