import crypta.lib.python.bt.conf.conf as conf

import yt.wrapper as yt
from library.python import resource

from crypta.graph.matching.direct.proto.types_pb2 import (
    TTypesInfo,
    TDirectEdge,
)
from crypta.graph.matching.direct.proto.public_edges_pb2 import TPublicEdges

from crypta.graph.soup.config.python import ID_TYPE
from crypta.lib.python.yt import schema_utils

import google.protobuf.text_format as tf


def get_public_edges():
    """ Get public edges from proto config """
    container = TPublicEdges()
    tf.Merge(resource.find('/public_edges.pb.txt'), container)
    for edge in container.PublicEdges:
        for destination in edge.DestinationTypes:
            yield (
                ID_TYPE.by_type(edge.SourceType).Name,
                ID_TYPE.by_type(destination).Name, )


def create_sorted_tables(yt_client, types_info, logger, generate_date, run_date, acl):
    for types_pair in types_info.types_pair:
        logger.info('Creating table ' + types_pair.path)

        yt_client.create(
            'map_node',
            yt.ypath_dirname(types_pair.path),
            ignore_existing=True,
            recursive=True
        )

        attributes = {
            'schema': get_schema_with_sort(),
            'optimize_for': 'scan',
            'generate_date': generate_date,
            'run_date': run_date,
        }

        table_acl = acl.get_pair_acl(types_pair.type_in, types_pair.type_out)
        if table_acl and conf.environment.mode != 'develop':
            attributes.update({
                'acl': table_acl,
                'inherit_acl': False,
            })
        logger.info('Type pair %s, acl = %s', types_pair, attributes.get('acl'))

        yt_client.create(
            'table',
            types_pair.path,
            attributes=attributes,
            force=True,
            recursive=True
        )


def _public_edges_generator(predicate):
    return filter(predicate, get_public_edges())


def collect_all_public_edges_without_indexes():
    types_info = TTypesInfo()

    for type_in, type_out in get_public_edges():
        types_pair = types_info.types_pair.add()
        types_pair.type_in = type_in
        types_pair.type_out = type_out

    return types_info


def _public_types_info_with_path(predicate, path_join):
    """ Make edge type info and path """
    types_info = TTypesInfo()
    paths = []
    index = 0
    for type_in, type_out in _public_edges_generator(predicate):
        types_pair = types_info.types_pair.add()
        types_pair.type_in = type_in
        types_pair.type_out = type_out
        types_pair.path = path_join(type_in, type_out)
        types_pair.index = index
        index += 1
        paths.append(types_pair.path)
    return types_info, paths


def get_types_info_with_paths_with_cryptaid(base_path):
    """ Only CryptaID edges """
    return _public_types_info_with_path(
        lambda pair: ID_TYPE.CRYPTA_ID.Name in pair,
        # /.../path/to/id_type/{type_in}/crypta_id
        # /.../path/to/id_type/crypta_id/{type_out}
        lambda type_in, type_out: yt.ypath_join(base_path, type_in, type_out),
    )


def get_types_info_with_paths_without_cryptaid(base_path):
    """ All except CryptaID edges """
    return _public_types_info_with_path(
        lambda pair: ID_TYPE.CRYPTA_ID.Name not in pair,
        # /.../path/to/id_type/{type_in}/direct/{type_out}
        lambda type_in, type_out: yt.ypath_join(base_path, type_in, 'direct', type_out),
    )


def get_schema():
    return schema_utils.get_schema_from_proto(TDirectEdge)


def get_schema_for_yuids():
    return get_schema_from_proto(TDirectEdge, excepted_columns=("target_id_type",))


def get_schema_with_sort():
    return get_schema_from_proto(TDirectEdge, sorted_by=["id", "id_type"])


def get_schema_from_proto(proto, sorted_by=(), excepted_columns=()):
    schema = schema_utils.get_schema_from_proto(proto)
    sorted_columns = list(sorted_by)
    new_schema = []
    for name in sorted_columns:
        for item in schema:
            if item["name"] == name:
                item['sort_order'] = 'ascending'
                new_schema.append(item)
    filtered_set = set(excepted_columns).union(sorted_columns)
    for item in schema:
        if item["name"] not in filtered_set:
            new_schema.append(item)
    return new_schema
