package ru.yandex.crypta.graph2.matching.human;

import java.time.LocalDate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.matching.human.config.HumanMatchingConfig;
import ru.yandex.crypta.graph2.matching.human.paths.ComponentTableWithEdgesBetween;
import ru.yandex.crypta.graph2.matching.human.paths.ComponentsRawTablesWithEdgesBetween;
import ru.yandex.crypta.graph2.matching.human.paths.SoupTables;
import ru.yandex.crypta.graph2.matching.human.strategy.HumanMatchingStrategyProvider;
import ru.yandex.crypta.graph2.matching.human.workflow.component.ProcessComponentsTask;
import ru.yandex.crypta.graph2.matching.human.workflow.component.ProcessComponentsTask.ProcessComponentTables;
import ru.yandex.crypta.graph2.matching.human.workflow.init.InitComponentsAndNeighboursTask;
import ru.yandex.crypta.graph2.matching.human.workflow.init.InitComponentsAndNeighboursTask.GraphInitTables;
import ru.yandex.crypta.graph2.matching.human.workflow.init.InitComponentsAndNeighboursTask.MatchingInputTables;
import ru.yandex.crypta.graph2.matching.human.workflow.merge.MergeConsensusTask;
import ru.yandex.crypta.graph2.matching.human.workflow.merge.MergeOffersTables;
import ru.yandex.crypta.graph2.matching.human.workflow.merge.UseMergeStrategyTask;
import ru.yandex.crypta.graph2.matching.human.workflow.merge_apply.ApplyMergeTask;
import ru.yandex.crypta.graph2.matching.human.workflow.merge_apply.ApplyMergeTask.ReadyForMergeTables;
import ru.yandex.crypta.graph2.matching.human.workflow.neighbours.PostProcessMergeOffers;
import ru.yandex.crypta.graph2.matching.human.workflow.neighbours.PostProcessMergeOffers.NeighboursInfo;
import ru.yandex.crypta.graph2.matching.human.workflow.output.CopyToOutputTask;
import ru.yandex.crypta.graph2.matching.human.workflow.output.CopyToOutputTask.CopyToOutputTables;
import ru.yandex.crypta.graph2.matching.human.workflow.output.JoinHeavyColumnsBackTask;
import ru.yandex.crypta.graph2.matching.human.workflow.prepare.PrepareComponentsForMergeTask;
import ru.yandex.crypta.graph2.matching.human.workflow.prepare.PrepareComponentsForMergeTask.ComponentsByMergeKeyTables;
import ru.yandex.inside.yt.kosher.cypress.YPath;

public class HumanGraphMatchWorkflow {

    private static final Logger LOG = LoggerFactory.getLogger(HumanGraphMatchWorkflow.class);

    private final Dao dao;


    private final InitComponentsAndNeighboursTask initComponentsAndNeighboursTask;
    private final PrepareComponentsForMergeTask prepareComponentsForMergeTask;
    private final UseMergeStrategyTask useMergeStrategyTask;
    private final MergeConsensusTask mergeConsensusTask;
    private final ApplyMergeTask applyMergeTask;
    private final PostProcessMergeOffers postProcessMergeOffers;
    private final ProcessComponentsTask processComponentsTask;

    private final JoinHeavyColumnsBackTask joinHeavyColumnsBackTask;
    private final CopyToOutputTask copyToOutputTask;

    private final boolean doCleanup;
    private final boolean joinHeavyColumnsToResultingTables;

    public HumanGraphMatchWorkflow(Dao dao,
                                   HumanMatchingStrategyProvider matchingStrategy,
                                   YPath workdir,
                                   YPath outDir,
                                   LocalDate soupGenerationDate,
                                   HumanMatchingConfig humanMatchingConfig,
                                   String experimentName
    ) {
        this.dao = dao;
        this.doCleanup = humanMatchingConfig.doCleanup;
        this.joinHeavyColumnsToResultingTables = humanMatchingConfig.joinHeavyColumnsToResultingTables;

        this.initComponentsAndNeighboursTask = new InitComponentsAndNeighboursTask(
                dao, workdir.child("init"),
                matchingStrategy,
                humanMatchingConfig.filterEdgeTypes,
                humanMatchingConfig.edgesPerSourceLimit,
                experimentName
        );
        this.prepareComponentsForMergeTask = new PrepareComponentsForMergeTask(
                dao, workdir.child("prepare"), matchingStrategy, experimentName
        );
        this.useMergeStrategyTask = new UseMergeStrategyTask(
                dao, workdir.child("merge"), matchingStrategy, experimentName
        );
        this.mergeConsensusTask = new MergeConsensusTask(
                dao, workdir.child("merge"), matchingStrategy, experimentName
        );
        this.applyMergeTask = new ApplyMergeTask(
                dao, workdir.child("merge_apply"), matchingStrategy, experimentName
        );
        this.postProcessMergeOffers = new PostProcessMergeOffers(
                dao, workdir.child("merge_post"), matchingStrategy, experimentName
        );
        this.processComponentsTask = new ProcessComponentsTask(
                dao, workdir.child("component"), matchingStrategy, experimentName,
                humanMatchingConfig.recsComponentLimit
        );
        this.joinHeavyColumnsBackTask = new JoinHeavyColumnsBackTask(
                dao, workdir.child("component"), humanMatchingConfig, experimentName
        );
        this.copyToOutputTask = new CopyToOutputTask(
                dao, outDir, matchingStrategy, soupGenerationDate, experimentName
        );
    }


    public void matchGraph(SoupTables soupTables, Option<YPath> verticesPrevIterMatching) {

        LOG.info("Start matching");

        initComponentsAndNeighboursTask.run(new MatchingInputTables(soupTables, verticesPrevIterMatching));
        GraphInitTables graphInitTables = initComponentsAndNeighboursTask.getOutput();

        prepareComponentsForMergeTask.run(graphInitTables);
        ComponentsByMergeKeyTables mergeCandidates = prepareComponentsForMergeTask.getOutput();

        cleanup(graphInitTables.graphTables.allTables());

        ComponentTableWithEdgesBetween tablesToMerge = new ComponentTableWithEdgesBetween(
                mergeCandidates.componentsToMerge, graphInitTables.edgesBetweenComponentsMergeKey
        );
        useMergeStrategyTask.run(tablesToMerge);
        MergeOffersTables mergeOffersBeforeConsensus = useMergeStrategyTask.getOutput();

        cleanup(mergeCandidates.componentsToMerge);

        mergeConsensusTask.run(mergeOffersBeforeConsensus.ok);
        MergeConsensusTask.MergeConsensusTables mergeConsensusResult = mergeConsensusTask.getOutput();
        YPath mergeDecisions = mergeConsensusResult.mergeDecisionsConfirmed;
        MergeOffersTables mergeOffersAfterConsensus = mergeConsensusResult.mergeOffersTables;

        cleanup(mergeOffersBeforeConsensus.ok);

        applyMergeTask.run(
                new ReadyForMergeTables(
                        new ComponentsRawTablesWithEdgesBetween(
                                mergeCandidates.componentsWithNeighbours,
                                graphInitTables.edgesBetweenComponentsMergeKey
                        ),
                        graphInitTables.mergeNeighbours,
                        mergeDecisions
                )
        );
        ComponentsRawTablesWithEdgesBetween mergeResult = applyMergeTask.getOutput();

        processComponentsTask.run(
                new ProcessComponentsTask.InTables(
                        mergeCandidates.componentsNoNeighbours,
                        mergeResult,
                        mergeOffersBeforeConsensus,
                        mergeOffersAfterConsensus)
        );
        ProcessComponentTables componentsProcessed = processComponentsTask.getOutput();

        postProcessMergeOffers.run(componentsProcessed);
        NeighboursInfo neighboursInfo = postProcessMergeOffers.getOutput();

//        cleanup(mergeDecisionsConfirmed, mergeOffersFailed);

        // Non-skippable part
        if (joinHeavyColumnsToResultingTables) {
            joinHeavyColumnsBackTask.run(componentsProcessed.graphTables.allComponentTables);
        }

        copyToOutputTask.run(new CopyToOutputTables(
                componentsProcessed.graphTables.allComponentTables,
                componentsProcessed.componentsStats,
                neighboursInfo,
                componentsProcessed.mergeOffers,
                componentsProcessed.splitInfo,
                componentsProcessed.components
        ));

        cleanup(componentsProcessed.graphTables.allTables());
        cleanup(componentsProcessed.componentsStats);

        LOG.info("Done matching");

    }

    private void cleanup(CollectionF<YPath> tables) {
        if (doCleanup) {
            dao.ytCypress().removeAll(Cf.wrap(tables));
        }
    }

    private void cleanup(YPath... tables) {
        cleanup(Cf.wrap(tables));
    }

}
