package ru.yandex.crypta.graph2.matching.human;

import java.text.MessageFormat;
import java.time.LocalDate;
import java.util.Arrays;

import joptsimple.OptionParser;
import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.dao.yql.Yql;
import ru.yandex.crypta.graph2.dao.yql.YqlConfig;
import ru.yandex.crypta.graph2.dao.yt.YtConfig;
import ru.yandex.crypta.graph2.dao.yt.ops.YtOpsParams;
import ru.yandex.crypta.graph2.matching.human.config.ConfigParser;
import ru.yandex.crypta.graph2.matching.human.config.HumanMatchingConfig;
import ru.yandex.crypta.graph2.matching.human.config.HumanMatchingStrategyFactory;
import ru.yandex.crypta.graph2.matching.human.paths.SoupTables;
import ru.yandex.crypta.graph2.matching.human.strategy.HumanMatchingStrategyProvider;
import ru.yandex.crypta.graph2.utils.NativeLibHelper;
import ru.yandex.crypta.graph2.utils.YamlConfig;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;

import static ru.yandex.crypta.graph2.dao.yt.YtCypressHelper.GENERATION_DATE_ATTR;

public class HumanMatchingMain {

    private static final Logger LOG = LoggerFactory.getLogger(HumanMatchingMain.class);

    public static void main(String[] args) {
        NativeLibHelper.setLocalJavaLibraryPath();
        System.out.println(Arrays.asList(args));
        OptionParser parser = new OptionParser();

        OptionSpec<String> configArg = parser
                .accepts("config")
                .withRequiredArg();

        OptionSpec<Boolean> copyToDictArg = parser
                .accepts("copyToDict")
                .withRequiredArg()
                .ofType(Boolean.class)
                .defaultsTo(false);

        OptionSpec<Integer> iterationIdArg = parser.accepts("iterationId")
                .withRequiredArg()
                .ofType(Integer.class)
                .defaultsTo(0);

        OptionSpec<Boolean> isExperimentArg = parser
                .accepts("isExperiment")
                .withOptionalArg()
                .ofType(Boolean.class)
                .defaultsTo(false);

        OptionSpec<String> experimentNameArg = parser
                .accepts("experimentName")
                .withOptionalArg()
                .defaultsTo("");

        OptionSpec<String> runIdArg = parser
                .accepts("runId")
                .withOptionalArg()
                .defaultsTo("");

        parser.allowsUnrecognizedOptions();
        OptionSet options = parser.parse(args);

        String configFile = options.valueOf(configArg);
        int iterationId = options.valueOf(iterationIdArg);
        boolean copyToDict = options.valueOf(copyToDictArg);
        boolean isExperiment = options.valueOf(isExperimentArg);
        String experimentName = options.valueOf(experimentNameArg);
        String runId = options.valueOf(runIdArg);

        if (isExperiment && experimentName.isEmpty()) {
            throw new IllegalArgumentException("ExperimentName can not be empty");
        }

        LOG.info("Start");

        Dao dao = createHttpBasedYtClientFromEnvVars();

        HumanMatchingConfig config = YamlConfig.readConfigFromFile(configFile, HumanMatchingConfig.class);
        LOG.info("Config content:\n{}", config);

        HumanMatchingMain humanMatchingMain = new HumanMatchingMain();
        humanMatchingMain.run(dao, config, copyToDict, iterationId, experimentName, runId);

    }

    public static Dao createHttpBasedYtClientFromEnvVars() {
        Yt yt = YtConfig.getYt(true);
        Yql yql = YqlConfig.getYql();

        return new Dao(yt, yql, new YtOpsParams());

    }

    public void runWithSoupTables(Dao dao, HumanMatchingConfig config, boolean copyToDict, int iterationId,
                                  SoupTables soupTables, String experimentName, String runId) {
        LocalDate soupGenerationDate = getSoupGenerationDate(dao, soupTables);

        HumanMatchingStrategyFactory humanMatchingStrategyFactory = new HumanMatchingStrategyFactory();
        HumanMatchingStrategyProvider matchingStrategyProvider = humanMatchingStrategyFactory.parseMatchingStrategy(config);
        if (runId != null && runId.length() > 0) {
            matchingStrategyProvider.setRunId(runId);
        }
        LOG.info("Run ID: {}\n", matchingStrategyProvider.getRunId());
        LOG.info("Soup generation date: {}\n", soupGenerationDate.toString());
        Option<YPath> prevIterVerticesMatching = dao.ytCypress().checkExistence(
                YPath.simple(config.prevMatchingTable)
        );

        logMatchingRun(iterationId, soupTables, matchingStrategyProvider, prevIterVerticesMatching);

        YPath workdir = YPath.simple(config.workdir);
        YPath outputDir = YPath.simple(config.outputDir);
        YPath dictDir = YPath.simple(config.dictDir);

        HumanGraphMatchWorkflow humanGraphMatchWorkflow = new HumanGraphMatchWorkflow(
                dao,
                matchingStrategyProvider,
                workdir,
                outputDir,
                soupGenerationDate,
                config,
                experimentName
        );
        humanGraphMatchWorkflow.matchGraph(soupTables, prevIterVerticesMatching);

        if (copyToDict) {
            // TODO: maybe links?
            dao.ytCypress().copyDirContent(outputDir, dictDir);
        }
    }

    public void runWithSoupTables(Dao dao, HumanMatchingConfig config, boolean copyToDict, int iterationId,
                                  SoupTables soupTables, String runId) {
        runWithSoupTables(dao, config, copyToDict, iterationId, soupTables, "", runId);
    }

    public void run(Dao dao, HumanMatchingConfig config, boolean copyToDict, int iterationId, String experimentName, String runId) {
        SoupTables soupTables = ConfigParser.parseSoupTables(config);

        runWithSoupTables(dao, config, copyToDict, iterationId, soupTables, experimentName, runId);
    }

    public void run(Dao dao, HumanMatchingConfig config, boolean copyToDict, int iterationId) {
        run(dao, config, copyToDict, iterationId, "", "");
    }

    protected LocalDate getSoupGenerationDate(Dao dao, SoupTables soupTables) {
        YPath tableWithAttr = soupTables.mergedSoupEdges;
        Option<LocalDate> generationDate = dao.ytCypress().getGenerationDate(tableWithAttr);
        return generationDate.getOrElse(() -> {
            LOG.warn("Param \"{}\" wasn't set in {}", GENERATION_DATE_ATTR, tableWithAttr);
            return LocalDate.now();
        });
    }

    protected void logMatchingRun(int iterationId,
                                SoupTables preparedSoup,
                                HumanMatchingStrategyProvider matchingStrategy,
                                Option<YPath> prevIterVerticesMatching) {
        LocalDate startDate = LocalDate.now();
        String prevMatchingMessage;
        if (prevIterVerticesMatching.isPresent()) {
            prevMatchingMessage = MessageFormat.format(
                    "starting from existing matching {0}", prevIterVerticesMatching.get());
        } else {
            prevMatchingMessage = "starting from scratch";
        }

        LOG.info("\n=====================================\n" +
                        "> Running matching graph experiment \n" +
                        "> for on {}\n" +
                        "> in {} iteration\n" +
                        "> with experiment params:\n{}\n" +
                        "> using prepared soup:\n{}\n" +
                        "> {}" +
                        "\n=====================================\n",
                startDate, iterationId, matchingStrategy, preparedSoup, prevMatchingMessage);
    }

}
