package ru.yandex.crypta.graph2.matching.human;

import java.io.IOException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;
import joptsimple.OptionParser;
import joptsimple.OptionSet;
import joptsimple.OptionSpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.dao.yt.local.fastyt.LocalYtFactory;
import ru.yandex.crypta.graph2.dao.yt.local.fastyt.testdata.DataHelper;
import ru.yandex.crypta.graph2.matching.human.config.HumanMatchingConfig;
import ru.yandex.crypta.graph2.matching.human.helper.LocalSoupHelper;
import ru.yandex.crypta.graph2.matching.human.helper.SoupId;
import ru.yandex.crypta.graph2.matching.human.helper.SoupIdList;
import ru.yandex.crypta.graph2.matching.human.paths.SoupTables;
import ru.yandex.crypta.graph2.model.matching.component.ComponentCenter;
import ru.yandex.crypta.graph2.model.soup.vertex.Vertex;
import ru.yandex.crypta.graph2.utils.YamlConfig;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtImpl;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.misc.lang.DefaultToString;

import static ru.yandex.crypta.graph2.model.matching.component.ComponentCenter.CC_ID_COLUMN;
import static ru.yandex.crypta.graph2.model.matching.component.ComponentCenter.CC_ID_TYPE_COLUMN;

public class HumanMatchingMainLocalCustom {

    private static final Logger LOG = LoggerFactory.getLogger(HumanMatchingMainLocalRun.class);

    private static final YPath LOCAL_YT_GRAPH_ROOT = YPath.simple("//home/crypta/production/state/graph/v2");
    private static final YPath DATES_COUNT_PER_EDGE_TYPE_TABLE = LOCAL_YT_GRAPH_ROOT.child(
                "soup/cooked/stats/dates_count_per_edge_type"
    );

    private static final String VERTICES_NO_MULTIPROFILE_TABLE_NAME = "vertices_no_multi_profile";
    private static final String EDGES_BY_CRYPTA_ID_TABLE_NAME = "edges_by_crypta_id";
    private static final String CRYPTA_COMPONENTS_NEIGHBOURS_TABLE_NAME = "crypta_components_neighbours";
    private static final String CRYPTA_COMPONENTS_STATS_TABLE_NAME = "crypta_components_stats";
    private static final String VERTICES_PROPERTIES_BY_CRYPTA_ID_TABLE_NAME = "vertices_properties_by_crypta_id";
    private static final String MERGE_OFFERS_TABLE_NAME = "merge_offers";
    private static final String EDGES_BETWEEN_COMPONENTS_TABLE_NAME = "edges_between_components";
    private static final String EDGES_BETWEEN_COMPONENTS2_TABLE_NAME = "edges_between_components_by_crypta_id";

    private static YPath output;

    public static void main(String[] args) {

        OptionParser parser = new OptionParser();
        OptionSpec<String> configArg = parser
                .accepts("config")
                .withRequiredArg()
                .defaultsTo("config-local.yaml");

        OptionSpec<String> soupArg = parser.accepts("soup")
                .withOptionalArg();

        OptionSpec<String> idValueArg = parser
                .accepts("idValue")
                .requiredIf("soup")
                .withRequiredArg();

        OptionSpec<String> idTypeArg = parser
                .accepts("idType")
                .requiredIf("idValue").withRequiredArg();

        OptionSpec<String> iterationsArg = parser
                .accepts("iterations")
                .requiredIf("soup").withOptionalArg()
                .defaultsTo("5");

        OptionSpec<String> vertexLimitArg = parser
                .accepts("vertexLimit").withOptionalArg()
                .defaultsTo("1000");

        OptionSpec<String> idsFileArg = parser
                .accepts("idList").withOptionalArg();

        parser.allowsUnrecognizedOptions();
        OptionSet options = parser.parse(args);

        String configFile = options.valueOf(configArg);
        String ifSoup = options.valueOf(soupArg);
        String idValue = options.valueOf(idValueArg);
        String idType = options.valueOf(idTypeArg);
        int iterations = Integer.valueOf(options.valueOf(iterationsArg));
        int vertexLimit = Integer.valueOf(options.valueOf(vertexLimitArg));
        String idsFile = options.valueOf(idsFileArg);

        LOG.info(options.asMap().toString() +
                "\nConfig file:  " + configFile +
                "\nSoup:         " + ifSoup +
                "\nId Value:     " + idValue +
                "\nId Type:      " + idType +
                "\nIterations:   " + iterations +
                "\nVertex Limit: " + vertexLimit +
                "\nIds file:     " + idsFile);

        HumanMatchingConfig config = YamlConfig.readConfigFromFile(configFile, HumanMatchingConfig.class);
        LOG.info("Config content:\n{}", config);

        List<SoupId> idsToProcess = new ArrayList<>();
        if (idValue != null && idType != null) {
            idsToProcess.add(new SoupId(idValue, idType));
        } else if (idsFile != null) {
            ObjectMapper mapper = new ObjectMapper(new YAMLFactory());
            try {
                SoupIdList idList = mapper.readValue(Paths.get(idsFile).toFile(), SoupIdList.class);
                idsToProcess.addAll(idList.getIds());

                LOG.info("Identifier list:\n{}", idList);
            } catch (IOException e) {
                LOG.error("Error reading id list file.");
                e.printStackTrace();
                System.exit(1);
            }
        }

        YPath soupVerticesPropertiesTable = YPath.simple(config.soupVerticesPropertiesTable);
        YPath soupVerticesPropertiesDynamicTable = YPath.simple(config.soupVerticesPropertiesDynamicTable);

        // TODO: implement processing with previous matching table
        // YPath prevMatchingTable = YPath.simple(config.prevMatchingTable);

        output = YPath.simple(config.outputDir);


        Dao realDao = HumanMatchingMain.createHttpBasedYtClientFromEnvVars();
        Dao localDao = LocalYtFactory.createFileBasedLocalDao(Paths.get(config.localYtDir));
        HumanMatchingMain humanMatchingMainLocal = new HumanMatchingMain();

        if (!idsToProcess.isEmpty()) {
            LocalSoupHelper localSoupHelper = new LocalSoupHelper(localDao);

            YPath soupEdges = YPath.simple(config.soupEdgesBackwardsTable);

            localSoupHelper.dumpSoupByIds(
                    soupEdges.toString(), soupVerticesPropertiesDynamicTable.toString(),
                    config.soupEdgesTable, soupVerticesPropertiesTable.toString(), idsToProcess, vertexLimit
            );

            prepareTestData(realDao, localDao);

            SoupTables soupTables = new SoupTables(
                    YPath.simple(config.soupEdgesTable + "_chunk"),
                    YPath.simple(config.soupVerticesPropertiesTable + "_chunk"),
                    Option.of(YPath.simple(config.soupEdgeMessagesTable + "_chunk"))
            );

            ListF<CountStats> iterationStats = Cf.arrayList();
            for (int iter = 0; iter < iterations; iter++) {
                humanMatchingMainLocal.runWithSoupTables(localDao, config, false, iter, soupTables, String.valueOf(iter));
                YPath outputDir = YPath.simple(config.outputDir);

                CountStats countStats = new CountStats();
                countStats.componentsCount = countTableRecs(outputDir.child(CRYPTA_COMPONENTS_STATS_TABLE_NAME), localDao);
                countStats.verticesCount = countTableRecs(outputDir.child(VERTICES_NO_MULTIPROFILE_TABLE_NAME), localDao);
                countStats.edgesCount = countTableRecs(outputDir.child(EDGES_BY_CRYPTA_ID_TABLE_NAME), localDao);
                countStats.edgesBetweenCount = countTableRecs(outputDir.child(EDGES_BETWEEN_COMPONENTS_TABLE_NAME), localDao);
                iterationStats.add(countStats);
            }

            LOG.info("Components merging progress");
            for (int iter = 0; iter < iterations; iter++) {
                LOG.info("Iter " + iter + ", stats: " + iterationStats.get(iter));
            }
        } else {
            prepareTestData(realDao, localDao);
            humanMatchingMainLocal.run(localDao, config, false, 0);
        }

        uploadMatchedData(localDao, realDao);
    }

    private static int countTableRecs(YPath table, Dao localDao) {
        return localDao.yt().tables().read(table, YTableEntryTypes.YSON,
                i -> { return Cf.x(i).count(); }
        );
    }

    private static class CountStats extends DefaultToString {
        int componentsCount;
        int verticesCount;
        int edgesCount;
        int edgesBetweenCount;
    }

    private static void prepareTestData(Dao realDao, Dao localDao) {
        DataHelper dataHelper = new DataHelper((YtImpl) realDao.yt(), localDao.yt());
        dataHelper.dumpTableToFile(DATES_COUNT_PER_EDGE_TYPE_TABLE);
    }

    private static void uploadMatchedData(Dao localDao, Dao realDao) {
        LOG.info("Uploading matched data...");
        DataHelper dataHelper = new DataHelper((YtImpl) realDao.yt(), localDao.yt());

        YPath localBasePath = output;
        YPath uploadBasePath = output;

        for (String table : Cf.list(
                VERTICES_NO_MULTIPROFILE_TABLE_NAME,
                EDGES_BY_CRYPTA_ID_TABLE_NAME,
                CRYPTA_COMPONENTS_NEIGHBOURS_TABLE_NAME,
                CRYPTA_COMPONENTS_STATS_TABLE_NAME,
                VERTICES_PROPERTIES_BY_CRYPTA_ID_TABLE_NAME,
                MERGE_OFFERS_TABLE_NAME,
                EDGES_BETWEEN_COMPONENTS_TABLE_NAME,
                EDGES_BETWEEN_COMPONENTS2_TABLE_NAME)
        ) {
            dataHelper.uploadLocalTableToYt(
                    localBasePath.child(table),
                    uploadBasePath.child(table)
            );
        }
        LOG.info("Uploading matched data done.");

        realDao.ytOps().sortSync(uploadBasePath.child(VERTICES_NO_MULTIPROFILE_TABLE_NAME), Vertex.VERTEX_REDUCE_KEY);
        realDao.ytOps().sortSync(uploadBasePath.child(EDGES_BY_CRYPTA_ID_TABLE_NAME), ComponentCenter.CRYPTA_ID_KEY);
        realDao.ytOps().sortSync(uploadBasePath.child(CRYPTA_COMPONENTS_NEIGHBOURS_TABLE_NAME), Cf.list(CC_ID_COLUMN, CC_ID_TYPE_COLUMN));
        realDao.ytOps().sortSync(uploadBasePath.child(CRYPTA_COMPONENTS_STATS_TABLE_NAME), Cf.list("cryptaId"));
        realDao.ytOps().sortSync(uploadBasePath.child(VERTICES_PROPERTIES_BY_CRYPTA_ID_TABLE_NAME), Cf.list("cryptaId"));
        realDao.ytOps().sortSync(uploadBasePath.child(MERGE_OFFERS_TABLE_NAME), Cf.list("cryptaId"));
        realDao.ytOps().sortSync(
                uploadBasePath.child(EDGES_BETWEEN_COMPONENTS_TABLE_NAME),
                Cf.list("id1Type", "id2Type", "sourceType", "logSource", "id1", "id2")
        );
        realDao.ytOps().sortSync(
                uploadBasePath.child(EDGES_BETWEEN_COMPONENTS2_TABLE_NAME),
                Cf.list("cryptaId", "id1Type", "id2Type", "sourceType", "logSource", "id1", "id2")
        );
        LOG.info("\nSorting matched data done.");

        System.exit(0);
    }

}
