package ru.yandex.crypta.graph2.matching.human.config;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.crypta.graph.engine.proto.TStatsOptions;
import ru.yandex.crypta.graph2.matching.human.strategy.HumanMatchingStrategyProvider;
import ru.yandex.crypta.graph2.model.matching.component.score.ComponentScoringStrategy;
import ru.yandex.crypta.graph2.model.matching.component.score.HumanMultiHistogramScoringExperimental;
import ru.yandex.crypta.graph2.model.matching.component.score.HumanMultiHistogramScoringStrategy;
import ru.yandex.crypta.graph2.model.matching.component.score.NativeHumanScoringStrategy;
import ru.yandex.crypta.graph2.model.matching.component.score.SimpleStupidScoringStrategy;
import ru.yandex.crypta.graph2.model.matching.component.similarity.ComponentsExactSocdemSimilarityStrategy;
import ru.yandex.crypta.graph2.model.matching.component.similarity.ComponentsProbSocdemSimilarityStrategy;
import ru.yandex.crypta.graph2.model.matching.component.similarity.ComponentsRegionSimilarityStrategy;
import ru.yandex.crypta.graph2.model.matching.component.similarity.ComponentsSimilarityMultiStrategy;
import ru.yandex.crypta.graph2.model.matching.component.similarity.ComponentsSimilarityStrategy;
import ru.yandex.crypta.graph2.model.matching.graph.cryptaid.CryptaIdDispenser;
import ru.yandex.crypta.graph2.model.matching.graph.cryptaid.CryptaIdDispenserByNeighboursWeight;
import ru.yandex.crypta.graph2.model.matching.graph.cryptaid.CryptaIdDispenserWithPuidPriority;
import ru.yandex.crypta.graph2.model.matching.merge.MergeKeyType;
import ru.yandex.crypta.graph2.model.matching.merge.MergeOfferPriority;
import ru.yandex.crypta.graph2.model.matching.merge.algo.merge.MergeAlgorithm;
import ru.yandex.crypta.graph2.model.matching.merge.algo.merge.MergeByScoreAndSimilarityAlgorithm;
import ru.yandex.crypta.graph2.model.matching.merge.algo.score.WeightedLinkScoringStrategy;
import ru.yandex.crypta.graph2.model.matching.merge.algo.split.DummySplitAlgorithm;
import ru.yandex.crypta.graph2.model.matching.merge.algo.split.MaxMinSplitAlgorithm;
import ru.yandex.crypta.graph2.model.matching.merge.algo.split.SplitAlgorithm;
import ru.yandex.crypta.graph2.model.matching.merge.algo.split.SplitByMinEdgeAlgorithm;
import ru.yandex.crypta.graph2.model.soup.edge.weight.DefaultEdgeInfoProvider;
import ru.yandex.crypta.graph2.model.soup.edge.weight.EdgeInfoProvider;
import ru.yandex.crypta.graph2.model.soup.edge.weight.SurvivalEdgeInfoProvider;
import ru.yandex.misc.lang.DefaultToString;

public class HumanMatchingStrategyFactory extends DefaultToString {

    private static final SetF<String> ALLOWED_MERGE_TYPES_CODES = Cf.set(
            MergeKeyType.BY_EDGE.name(),
            MergeKeyType.BY_COMPONENT.name(),
            MergeKeyType.BY_EDGES_CUT.name()
    );

    private ComponentScoringStrategy createComponentScoringStrategy(HumanMatchingConfig config) {
        Option<Double> crossDeviceWeight = Option.of(7.5);
        TStatsOptions scoreOptions = TStatsOptions
                .newBuilder()
                .setCrossDeviceWeight(config.nativeScoreOptions.crossDeviceWeight)
                .setSocdemWeight(config.nativeScoreOptions.socdemWeight)
                .build();

        String name = config.componentScoringStrategy;
        if ("human_histogram".equals(name)) {
            return new HumanMultiHistogramScoringStrategy(false, crossDeviceWeight);
        } else if ("human_histogram_shards_opts".equals(name)) {
            // merge all inactive logins to active account
            return new HumanMultiHistogramScoringStrategy(true, crossDeviceWeight);
        } else if ("native_human_histogram".equals(name)) {
            return new NativeHumanScoringStrategy(scoreOptions);
        } else if ("exp_native_human_histogram".equals(name)) {
            return new NativeHumanScoringStrategy(scoreOptions, true);
        }  else if ("simple".equals(name)) {
            return new SimpleStupidScoringStrategy();
        } else {
            throw new IllegalArgumentException(String.format("No ComponentScoringStrategy found for \"%s\"", name));
        }
    }

    private ComponentsSimilarityStrategy createComponentsSimilarityStrategy(String name) {
        if ("exact_socdem".equals(name)) {
            return new ComponentsExactSocdemSimilarityStrategy();
        } else if ("prob_socdem".equals(name)) {
            return new ComponentsProbSocdemSimilarityStrategy();
        } else if ("region".equals(name)) {
            return new ComponentsRegionSimilarityStrategy();
        } else {
            throw new IllegalArgumentException(String.format("No ComponentSimilarityStrategy found for \"%s\"", name));
        }
    }

    private ComponentsSimilarityMultiStrategy createComponentSimilarityStrategy(HumanMatchingConfig config) {
        ListF<ComponentsSimilarityStrategy> similarityStrategies = Cf.wrap(config.componentsSimilarityStrategies).map(
                this::createComponentsSimilarityStrategy
        ).plus(Cf.wrap(config.weakComponentsSimilarityStrategies).map(
                name->createComponentsSimilarityStrategy(name).asWeak()
        ));
        return new ComponentsSimilarityMultiStrategy(similarityStrategies);
    }


    private MergeAlgorithm createMergeAlgorithm(HumanMatchingConfig config, EdgeInfoProvider edgeInfoProvider) {
        String name = config.mergeStrategy;
        MergeByScoreAndSimilarityAlgorithm.MergeMode mergeMode = new MergeByScoreAndSimilarityAlgorithm.MergeMode();
        if (name != null) {
            if (name.contains("force_ignore_score")) {
                mergeMode.forceIgnoreScores = true;
            } else if (name.contains("weak_ignore_score")) {
                mergeMode.ignoreScoresForGoodLinks = true;
            }
        }
        if (config.mergeSizeLimit > 0) {
            mergeMode.mergeSizeLimit = config.mergeSizeLimit;
        }
        return new MergeByScoreAndSimilarityAlgorithm(
                createComponentScoringStrategy(config),
                createComponentSimilarityStrategy(config),
                edgeInfoProvider,
                mergeMode
        );
    }

    private MergeAlgorithm createExperimentalMergeAlgorithm(HumanMatchingConfig config, EdgeInfoProvider edgeInfoProvider) {
        return new MergeByScoreAndSimilarityAlgorithm(
                new HumanMultiHistogramScoringExperimental(false, Option.of(7.5)),
                createComponentSimilarityStrategy(config),
                edgeInfoProvider
        );
    }

    private SplitAlgorithm createSplitAlgorithm(HumanMatchingConfig config) {
        String name = config.splitStrategy;

        ComponentScoringStrategy componentScoringStrategy =
                createComponentScoringStrategy(config);
        EdgeInfoProvider edgeInfoProvider = new SurvivalEdgeInfoProvider();

        if ("new_split".equals(name)) {
            return new MaxMinSplitAlgorithm(componentScoringStrategy, edgeInfoProvider,
                    new WeightedLinkScoringStrategy(
                            edgeInfoProvider,
                            500,
                            1000,
                            12,
                            4
                    )
            );
        } else if ("dummy".equals(name)) {
            return new DummySplitAlgorithm(componentScoringStrategy);
        } else if ("default".equals(name) || name == null) {
            return new SplitByMinEdgeAlgorithm(componentScoringStrategy, edgeInfoProvider);
        } else {
            throw new IllegalArgumentException(String.format("\"%s\" is wrong name for splitStrategy", name));
        }
    }

    private MergeKeyType createMergeType(String mergeType) {
        if (mergeType != null) {
            String mergeTypeStr = mergeType.toUpperCase();
            if (ALLOWED_MERGE_TYPES_CODES.containsTs(mergeTypeStr)) {
                return MergeKeyType.valueOf(mergeTypeStr);
            }
        }

        throw new IllegalArgumentException("Merge type is not supported: " + mergeType);

    }

    private CryptaIdDispenser createCryptaIdDispenser(String name, EdgeInfoProvider edgeInfoProvider) {
        if ("default".equals(name)) {
            return new CryptaIdDispenserByNeighboursWeight(edgeInfoProvider);
        } else if ("puid".equals(name)) {
            CryptaIdDispenserByNeighboursWeight defaultDispenser =
                    new CryptaIdDispenserByNeighboursWeight(edgeInfoProvider);
            return new CryptaIdDispenserWithPuidPriority(edgeInfoProvider, defaultDispenser);
        }

        throw new IllegalArgumentException("CryptaIdDispenser is not supported: " + name);
    }

    private EdgeInfoProvider createEdgeInfoProvider(HumanMatchingConfig config) {
        // TODO: CRYPTR-1315 change to survival
        if (config.mergeStrategy != null && config.mergeStrategy.contains("new_split")) {
            return new SurvivalEdgeInfoProvider();
        }
        return new DefaultEdgeInfoProvider();
    }

    private MergeOfferPriority createMergePriority(HumanMatchingConfig config) {
        return new MergeOfferPriority(config.forceMergeSingleVertexComponents);
    }

    public HumanMatchingStrategyProvider parseMatchingStrategy(HumanMatchingConfig config) {

        EdgeInfoProvider edgeInfoProvider = createEdgeInfoProvider(config);

        return new HumanMatchingStrategyProvider(
                edgeInfoProvider,
                createCryptaIdDispenser(config.cryptaIdDispenser, edgeInfoProvider),
                createComponentScoringStrategy(config),
                createMergeAlgorithm(config, edgeInfoProvider),
                createExperimentalMergeAlgorithm(config, edgeInfoProvider),
                createSplitAlgorithm(config),
                createMergePriority(config),
                createMergeType(config.mergeType),
                config.skipReadyTasks
        );
    }


}
