package ru.yandex.crypta.graph2.matching.human.helper;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.common.exception.Exceptions;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.dao.yt.YtRpcConfig;
import ru.yandex.crypta.graph2.model.soup.edge.MultiEdgeKey;
import ru.yandex.crypta.lib.proto.TYtConfig;
import ru.yandex.crypta.lib.yt.DefaultYtService;
import ru.yandex.crypta.lib.yt.YtService;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.tables.YTableEntryTypes;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.yt.ytclient.proxy.SelectRowsRequest;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

public class LocalSoupHelper {

    private static final Logger LOG = LoggerFactory.getLogger(LocalSoupHelper.class);
    private static final YtService ytRpc = new DefaultYtService(
            TYtConfig.newBuilder()
                    .setRpcUser(YtRpcConfig.getUser())
                    .setToken(YtRpcConfig.getToken())
                    .setProxy(YtRpcConfig.getClusterName())
                    .build()
    );

    private final Dao localDao;

    private static final Long LOG_RECORDS_CUTOFF = 1000L;

    public LocalSoupHelper(Dao localDao) {
        this.localDao = localDao;
    }

    private UnversionedRowset selectRowsByIdAndIdType(
            YtClient client, String srcTable, String idValue, String idType
    ) {
        return selectRows(client, srcTable, idValue, idType,
                "* FROM [%s] WHERE id1 = '%s' AND id1Type = '%s'");
    }

    private UnversionedRowset selectRows(
            YtClient client, String srcTable, String idValue, String idType, String requestTemplate
    ) {
        SelectRowsRequest request = SelectRowsRequest.of(String.format(
                requestTemplate, srcTable, idValue, idType
        ));
        request.setInputRowsLimit(10_000_000);
        request.setOutputRowsLimit(10_000_000);

        return client.selectRows(request).join();
    }

    public void dumpSoupByIds(
            String soupEdgesBackwardsSrc, String verticesPropertiesSrc,
            String extractedSoupDst, String verticesPropertiesDst,
            List<SoupId> idsToProcess, int vertexLimit) {

        for (SoupId id : idsToProcess) {
            Set<SimpleVertex> vertexSet = getVertices(
                    YPath.simple(soupEdgesBackwardsSrc),
                    id.getId(), id.getIdType(),
                    vertexLimit
            );

            LOG.info("Start reading edges for " + id.getId() + " " + id.getIdType());
            ListF<YTreeMapNode> edges = getRowsByVertices(vertexSet, soupEdgesBackwardsSrc);

            ListF<YTreeMapNode> verticesProperties = getVerticesProperties(vertexSet, verticesPropertiesSrc);

            // Dumping edges and vertices properties for particular id
            localDao.yt().tables().write(YPath.simple(extractedSoupDst + "_chunk").append(true),
                    YTableEntryTypes.YSON, edges);
            localDao.yt().tables().write(YPath.simple(verticesPropertiesDst + "_chunk").append(true),
                    YTableEntryTypes.YSON, verticesProperties);

            localDao.ytOps().sortSync(YPath.simple(extractedSoupDst + "_chunk"), MultiEdgeKey.MULTI_EDGE_UNIQUE_KEY);
            localDao.ytOps().sortSync(YPath.simple(verticesPropertiesDst + "_chunk"), Cf.list("id", "id_type"));
        }

        LOG.info("Dump done. Start matching...");
    }

    private Set<SimpleVertex> getVertices(YPath soupSrc, String idValue, String idType, int vertexLimit) {
        Set<SimpleVertex> vertexSet = new HashSet<>();
        Queue<SimpleVertex> vertexQueue = new LinkedList<>();

        SimpleVertex rootVertex = new SimpleVertex(idValue, idType);
        vertexQueue.add(rootVertex);
        vertexSet.add(rootVertex);

        YtClient client = ytRpc.getHahnRpc();


        LOG.info("Reading initial bunch of vertices from " + soupSrc.toString());
        UnversionedRowset initialRowset = selectRowsByIdAndIdType(client, soupSrc.toString(), idValue, idType);

        fulfillStructuresFromRowset(vertexSet, vertexQueue, initialRowset);


        Map<String, Long> recCounter = new HashMap<>();
        recCounter.put("rows", 0L);


        while (!vertexQueue.isEmpty()) {
            SimpleVertex vertex = vertexQueue.poll();
            vertexSet.add(vertex);

            UnversionedRowset rowset = selectRowsByIdAndIdType(client, soupSrc.toString(), vertex.getIdValue(),
                    vertex.getIdType());
            recCounter.put("rows", recCounter.get("rows") + 1);

            if (recCounter.get("rows") % LOG_RECORDS_CUTOFF == 0L) {
                LOG.info("Read " + recCounter.get("rows") + " from " + soupSrc.toString());
            }

            fulfillStructuresFromRowset(vertexSet, vertexQueue, rowset);

            if (vertexSet.size() > vertexLimit) {
                LOG.info("Vertex set size reached determined limit of " + vertexLimit + ".");
                LOG.info("Finished reading.");
                break;
            }
        }


        return vertexSet;
    }

    private void fulfillStructuresFromRowset(Set<SimpleVertex> vertexSet, Queue<SimpleVertex> vertexQueue,
                                             UnversionedRowset rowset) {
        for (YTreeMapNode row : rowset.getYTreeRows()) {
            SimpleVertex child = new SimpleVertex(row.getString("id2"), row.getString("id2Type"));

            if (!vertexSet.contains(child)) {
                vertexQueue.add(child);
            }

            vertexSet.add(child);
        }
    }

    private ListF<YTreeMapNode> getVerticesProperties(Set<SimpleVertex> vertexSet, String srcTable) {
        LOG.info("Start reading vertices properties");
        ListF<YTreeMapNode> rows = Cf.arrayList();

        Map<String, Long> recCounter = new HashMap<>();
        recCounter.put("rows", 0L);

        YtClient client = ytRpc.getHahnRpc();

        for (SimpleVertex vertex : vertexSet) {
            UnversionedRowset vertexRowset = selectRows(client, srcTable, vertex.getIdValue(), vertex.getIdType(),
                    "* FROM [%s] WHERE id = '%s' AND id_type = '%s'");
            for (YTreeMapNode row : vertexRowset.getYTreeRows()) {
                rows.add(row);

                recCounter.put("rows", recCounter.get("rows") + 1);

                if (recCounter.get("rows") % 50 == 0) {
                    LOG.info("Read " + recCounter.get("rows") + " of vertices properties.");
                }
            }
        }


        LOG.info("Finished reading vertices properties");

        return rows;
    }

    private ListF<YTreeMapNode> getRowsByVertices(Set<SimpleVertex> vertexSet, String srcTable) {
        LOG.info("Start reading records for selected vertices.");
        ListF<YTreeMapNode> rowList = Cf.arrayList();
        ExecutorService executor = Executors.newFixedThreadPool(16);

        Map<String, Long> recCounter = new HashMap<>();
        recCounter.put("rows", 0L);

        LOG.info("Read rows from " + srcTable);

        YtClient client = ytRpc.getHahnRpc();

        ListF<CompletableFuture> futures = Cf.arrayList();

        for (SimpleVertex vertex : vertexSet) {

            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> {
                UnversionedRowset rowset = selectRows(client, srcTable, vertex.getIdValue(), vertex.getIdType(),
                        "* FROM [%s] WHERE id1 = '%s' AND id1Type = '%s'");

                for (YTreeMapNode row : rowset.getYTreeRows()) {
                    rowList.add(row);

                    recCounter.put("rows", recCounter.get("rows") + 1);
                    if (recCounter.get("rows") % LOG_RECORDS_CUTOFF == 0) {
                        LOG.info("Read " + recCounter.get("rows") + " of bound vertices.");
                    }
                }

            }, executor);

            futures.add(future);
        }

        CompletableFuture<Void> combinedFuture = CompletableFuture.allOf(futures.toArray(CompletableFuture.class));

        try {
            combinedFuture.get();
        } catch (InterruptedException | ExecutionException e) {
            LOG.error("Can't get rows from " + srcTable);
            throw Exceptions.unavailable();
        }


        return rowList;
    }
}
