package ru.yandex.crypta.graph2.matching.human.workflow.component;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.dao.yt.TableWithTmp;
import ru.yandex.crypta.graph2.matching.human.helper.YQLTypeConfigGenerator;
import ru.yandex.crypta.graph2.matching.human.paths.ComponentsRawTables;
import ru.yandex.crypta.graph2.matching.human.paths.ComponentsRawTablesWithEdgesBetween;
import ru.yandex.crypta.graph2.matching.human.strategy.HumanMatchingStrategyProvider;
import ru.yandex.crypta.graph2.matching.human.workflow.HumanMatchingTask;
import ru.yandex.crypta.graph2.matching.human.workflow.component.ops.CalculateComponentInfoReducer;
import ru.yandex.crypta.graph2.matching.human.workflow.component.ops.JoinNewCryptaIdsWithEdgeBetweenComponentsReducer;
import ru.yandex.crypta.graph2.matching.human.workflow.component.ops.indevice.IndeviceLink;
import ru.yandex.crypta.graph2.matching.human.workflow.merge.MergeOffersTables;
import ru.yandex.crypta.graph2.matching.human.workflow.prepare.ConcatTablesHelper;
import ru.yandex.crypta.graph2.model.matching.component.ComponentCenter;
import ru.yandex.crypta.graph2.model.matching.component.ComponentStats;
import ru.yandex.crypta.graph2.model.matching.component.CryptaIdChangedEvent;
import ru.yandex.crypta.graph2.model.matching.edge.EdgeBetweenComponents;
import ru.yandex.crypta.graph2.model.matching.edge.EdgeBetweenWithNewCryptaIds;
import ru.yandex.crypta.graph2.model.matching.edge.EdgeInComponent;
import ru.yandex.crypta.graph2.model.matching.merge.MergeOffer;
import ru.yandex.crypta.graph2.model.matching.proto.SplitInfo;
import ru.yandex.crypta.graph2.model.matching.proto.TGraphRecord;
import ru.yandex.crypta.graph2.model.matching.vertex.VertexInComponent;
import ru.yandex.crypta.graph2.model.soup.edge.Edge;
import ru.yandex.crypta.graph2.model.soup.props.VertexPropertiesSchemaHelper;
import ru.yandex.crypta.graph2.workflow.IterableTables;
import ru.yandex.inside.yt.kosher.cypress.YPath;

public class ProcessComponentsTask
        extends HumanMatchingTask<ProcessComponentsTask.InTables, ProcessComponentsTask.ProcessComponentTables>
{

    private static final Logger LOG = LoggerFactory.getLogger(ProcessComponentsTask.class);

    private YPath finalVertices;
    private YPath finalVerticesProperties;
    private YPath finalEdges;
    private YPath betweenEdgesAfterCluster;

    private YPath componentsStats;
    private YPath mergeOffers;
    private YPath splitInfo;
    private YPath splitInfoNotSplitted;
    private YPath cryptaIdChangeStats;
    private YPath indeviceLinks;
    private YPath edgesBetweenWithNewCryptaIds;
    private YPath edgesBetween;
    private YPath components;

    private YPath oom;
    private YPath strange;

    private ConcatTablesHelper concatHelper;
    private int recsComponentLimit;

    public ProcessComponentsTask(Dao dao, YPath workdir, HumanMatchingStrategyProvider matchingStrategy, String nameSuffix, int recsComponentLimit) {
        super(dao, workdir, matchingStrategy, nameSuffix);

        this.finalVertices = workdir.child("vertices_final");
        this.finalVerticesProperties = workdir.child("vertices_properties_final");
        this.finalEdges = workdir.child("edges_final");
        this.betweenEdgesAfterCluster = workdir.child("edges_between_clustered");

        this.componentsStats = workdir.child("crypta_components_stats");
        this.mergeOffers = workdir.child("merge_offers_after_component_calc");
        this.splitInfo = workdir.child("split_info_by_final_cryptaid");
        this.splitInfoNotSplitted = workdir.child("split_info_not_splitted");
        this.cryptaIdChangeStats = workdir.child("crypta_id_change_stats");
        this.indeviceLinks = workdir.child("indevice_links");
        this.edgesBetweenWithNewCryptaIds = workdir.child("edges_between_with_new_cryptaids");
        this.edgesBetween = workdir.child("edges_between");
        this.components = workdir.child("components");

        this.oom = workdir.child("oom");
        this.strange = workdir.child("strange");
        this.concatHelper = new ConcatTablesHelper(dao);
        this.recsComponentLimit = recsComponentLimit;
    }

    private ComponentsRawTables prepareComponentsTablesForComponentsProcessing(ComponentsRawTables graphTablesNotMerge, ComponentsRawTablesWithEdgesBetween mergeResult) {
        dao.ytCypress().ensureDir(workdir);

        return concatHelper.concatComponents(
                Cf.list(graphTablesNotMerge, mergeResult.allComponentTables),
                new ComponentsRawTables(
                        workdir.child("vertices_tmp"),
                        workdir.child("vertices_properties_tmp"),
                        workdir.child("edges_tmp")
                )
        );

    }

    private YPath prepareMergeOffersTablesForComponentsProcessing(MergeOffersTables mergeOffers, MergeOffersTables finalMergeStatuses) {
        dao.ytCypress().ensureDir(workdir);

        return concatHelper.mergeTablesWithSchemaFromOutput(
                Cf.list(mergeOffers.failed, finalMergeStatuses.failed),
                workdir.child("unmerged_offers_tmp"),
                MergeOffer.class
        );
    }

    public static class InTables {
        public ComponentsRawTables graphTablesNotMerge;
        public ComponentsRawTablesWithEdgesBetween mergeResult;
        public MergeOffersTables mergeOffers;
        public MergeOffersTables finalMergeStatuses;

        public InTables(ComponentsRawTables graphTablesNotMerge, ComponentsRawTablesWithEdgesBetween mergeResult,
                MergeOffersTables mergeOffers, MergeOffersTables finalMergeStatuses) {
            this.graphTablesNotMerge = graphTablesNotMerge;
            this.mergeResult = mergeResult;
            this.mergeOffers = mergeOffers;
            this.finalMergeStatuses = finalMergeStatuses;
        }
    }

    @Override
    protected void runImpl(InTables in) {
        dao.ytCypress().ensureDir(workdir);
        ComponentsRawTables mergedTables = prepareComponentsTablesForComponentsProcessing(
                in.graphTablesNotMerge,
                in.mergeResult
        );
        YPath unmergedOffers = prepareMergeOffersTablesForComponentsProcessing(
                in.mergeOffers,
                in.finalMergeStatuses
        );
        YPath edgesBetweenIn = in.mergeResult.edgesBetweenComponents;

        LOG.info("Preparing for for CalculateComponentInfoReducer");

        // assume other tables are sorted
        dao.ytOps().sortAllParallel(Cf.list(unmergedOffers, edgesBetweenIn), ComponentCenter.CRYPTA_ID_KEY);

        dao.ytTr().withTransactionContext(dao, (txContext) -> {

            ListF<TableWithTmp> outTables = Cf.list(
                    txContext.createWithSchema(finalVertices, VertexInComponent.class),
                    txContext.createWithSchema(
                            finalVerticesProperties,
                            VertexPropertiesSchemaHelper.getUnionVertexPropertiesSchema()
                    ),
                    txContext.createWithSchema(finalEdges, EdgeInComponent.class),
                    txContext.createWithSchema(betweenEdgesAfterCluster, EdgeBetweenComponents.class),
                    txContext.createWithSchema(componentsStats, ComponentStats.class),
                    txContext.createWithSchema(mergeOffers, MergeOffer.class),
                    txContext.createWithSchema(splitInfo, SplitInfo.class),
                    txContext.createWithSchema(splitInfoNotSplitted, SplitInfo.class),
                    txContext.createWithSchema(cryptaIdChangeStats, CryptaIdChangedEvent.class),
                    txContext.createWithSchema(indeviceLinks, IndeviceLink.class),
                    txContext.createNoSchema(oom),
                    txContext.createNoSchema(strange),
                    txContext.createWithSchema(edgesBetweenWithNewCryptaIds, EdgeBetweenWithNewCryptaIds.class),
                    txContext.createWithSchema(components, TGraphRecord.class)
            );

            LOG.info("Running CalculateComponentInfoReducer");
            dao.ytOps().reduceSync(
                    txContext.getTransactionId(),
                    mergedTables.allTables().plus(unmergedOffers).plus(edgesBetweenIn),
                    outTables.map(TableWithTmp::getTmpTablePath),
                    ComponentCenter.CRYPTA_ID_KEY,
                    new CalculateComponentInfoReducer(
                            params.getEdgeInfoProvider(),
                            params.getComponentScoringStrategy(),
                            params.getCryptaIdDispenser(),
                            params.getSplitAlgorithm(),
                            recsComponentLimit
                    )
            );

            txContext.finalizeTables(outTables);
        });

        dao.ytOps().sortSync(edgesBetweenWithNewCryptaIds, Edge.EDGE_UNIQUE_KEY);
        dao.ytCypress().createTableWithSchema(edgesBetween, EdgeBetweenComponents.class);
        LOG.info("Running JoinEdgesBetweenComponentsWithNewCryptaIdsReducer");
        dao.ytOps().reduceSync(
                Cf.list(edgesBetweenWithNewCryptaIds),
                Cf.list(edgesBetween),
                Edge.EDGE_UNIQUE_KEY,
                new JoinNewCryptaIdsWithEdgeBetweenComponentsReducer()
        );
        setYQLProtoFields();
    }

    @Override
    public ProcessComponentTables getOutput() {
        ComponentsRawTablesWithEdgesBetween finalTables = new ComponentsRawTablesWithEdgesBetween(
                new ComponentsRawTables(
                        finalVertices,
                        finalVerticesProperties,
                        finalEdges
                ), betweenEdgesAfterCluster);
        return new ProcessComponentTables(
                finalTables,
                componentsStats,
                mergeOffers,
                splitInfo,
                splitInfoNotSplitted,
                cryptaIdChangeStats,
                edgesBetween,
                components
        );
    }

    private void setYQLProtoFields() {
        try {
            String yqlProtoField = YQLTypeConfigGenerator.getYQlProtoFieldForSplitInfo();
            dao.ytCypress().setAttribute(splitInfo, "_yql_proto_field_offers", yqlProtoField);
            dao.ytCypress().setAttribute(splitInfoNotSplitted, "_yql_proto_field_offers", yqlProtoField);
        } catch (java.lang.UnsatisfiedLinkError e) {
            LOG.info("Cannot find matching-model_java");
        }
    }

    @Override
    public String getDescription() {
        return "Calculates some info for each component";
    }

    public static class ProcessComponentTables extends IterableTables {
        public ComponentsRawTablesWithEdgesBetween graphTables;
        public YPath componentsStats;
        public YPath mergeOffers;
        public YPath changeCryptaIdStats;
        public YPath splitInfo;
        public YPath splitInfoFailed;
        public YPath edgesBetween;
        public YPath components;

        public ProcessComponentTables(ComponentsRawTablesWithEdgesBetween graphTables, YPath componentsStats,
                YPath mergeOffers, YPath splitInfo, YPath splitInfoFailed, YPath changeCryptaIdStats, YPath edgesBetween, YPath components) {
            this.graphTables = graphTables;
            this.componentsStats = componentsStats;
            this.mergeOffers = mergeOffers;
            this.changeCryptaIdStats = changeCryptaIdStats;
            this.splitInfo = splitInfo;
            this.splitInfoFailed = splitInfoFailed;
            this.edgesBetween = edgesBetween;
            this.components = components;
        }

        @Override
        public ListF<YPath> allTables() {
            return Cf.list(
                    componentsStats,
                    mergeOffers,
                    changeCryptaIdStats,
                    splitInfo,
                    splitInfoFailed,
                    edgesBetween
            ).plus(graphTables.allTables());
        }
    }

    public static class UnmergedOffersWithEdgesBetween {
        public YPath mergeOffers;
        public YPath edgesBetween;

        public UnmergedOffersWithEdgesBetween(YPath mergeOffers, YPath edgesBetween) {
            this.mergeOffers = mergeOffers;
            this.edgesBetween = edgesBetween;
        }
    }
}
