package ru.yandex.crypta.graph2.matching.human.workflow.component.ops;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.stream.Stream;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.crypta.graph2.dao.yt.bendable.YsonMultiEntityReducerWithKey;
import ru.yandex.crypta.graph2.matching.human.workflow.component.ops.indevice.IndeviceLink;
import ru.yandex.crypta.graph2.matching.human.workflow.component.ops.indevice.MarkOutIndeviceEdges;
import ru.yandex.crypta.graph2.matching.human.workflow.merge.ops.helper.GraphInfoRecsParser;
import ru.yandex.crypta.graph2.model.matching.component.Component;
import ru.yandex.crypta.graph2.model.matching.component.ComponentCenter;
import ru.yandex.crypta.graph2.model.matching.component.ComponentStats;
import ru.yandex.crypta.graph2.model.matching.component.CryptaIdChangedEvent;
import ru.yandex.crypta.graph2.model.matching.component.GraphInfo;
import ru.yandex.crypta.graph2.model.matching.component.GraphRecord;
import ru.yandex.crypta.graph2.model.matching.component.score.ComponentScoringStrategy;
import ru.yandex.crypta.graph2.model.matching.edge.EdgeBetweenComponents;
import ru.yandex.crypta.graph2.model.matching.edge.EdgeBetweenWithNewCryptaIds;
import ru.yandex.crypta.graph2.model.matching.edge.EdgeInComponent;
import ru.yandex.crypta.graph2.model.matching.graph.cryptaid.CryptaIdDispenser;
import ru.yandex.crypta.graph2.model.matching.merge.MergeKey;
import ru.yandex.crypta.graph2.model.matching.merge.MergeKeyWithNewCryptaIds;
import ru.yandex.crypta.graph2.model.matching.merge.MergeOffer;
import ru.yandex.crypta.graph2.model.matching.merge.MergeOfferStatus;
import ru.yandex.crypta.graph2.model.matching.merge.algo.split.SplitAlgorithm;
import ru.yandex.crypta.graph2.model.matching.merge.algo.split.inner.ComponentConnectivityInspector;
import ru.yandex.crypta.graph2.model.matching.merge.algo.split.inner.ComponentSplitDendrogram;
import ru.yandex.crypta.graph2.model.matching.proto.SplitInfo;
import ru.yandex.crypta.graph2.model.matching.score.MetricsTree;
import ru.yandex.crypta.graph2.model.matching.vertex.VertexInComponent;
import ru.yandex.crypta.graph2.model.soup.edge.Edge;
import ru.yandex.crypta.graph2.model.soup.edge.weight.EdgeInfoProvider;
import ru.yandex.crypta.graph2.model.soup.vertex.Vertex;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.operations.Statistics;
import ru.yandex.inside.yt.kosher.operations.Yield;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.misc.lang.Check;

public class CalculateComponentInfoReducer extends YsonMultiEntityReducerWithKey<String> {

    public static final int VERTICES_OUT_INDEX = 0;
    public static final int VERTICES_PROPERTIES_OUT_INDEX = 1;
    public static final int EDGES_OUT_INDEX = 2;
    public static final int EDGES_BETWEEN_SPLITTED_OUT_INDEX = 3;

    public static final int COMPONENT_STATS_INDEX = 4;
    public static final int NEIGHBOUR_MERGE_OFFERS_INDEX = 5;
    public static final int SPLIT_INFO_INDEX = 6;
    public static final int SPLIT_INFO_NOT_SPLITTED_INDEX = 7;
    public static final int CRYPTA_ID_CHANGE_STATS_INDEX = 8;
    public static final int INDEVICE_LINKS = 9;

    public static final int OOM_INDEX = 10;
    public static final int STRANGE_INDEX = 11;
    public static final int EDGES_BETWEEN = 12;
    public static final int COMPONENT_OUT_INDEX = 13;

    private final EdgeInfoProvider edgeInfoProvider;
    private final ComponentScoringStrategy componentScoringStrategy;
    private final SplitAlgorithm splitAlgorithm;
    private final CryptaIdDispenser cryptaIdDispenser;
    private int recsLimit = 10000;

    public CalculateComponentInfoReducer(
            EdgeInfoProvider edgeInfoProvider,
            ComponentScoringStrategy componentScoringStrategy,
            CryptaIdDispenser cryptaIdDispenser,
            SplitAlgorithm splitAlgorithm) {

        this.edgeInfoProvider = edgeInfoProvider;
        this.componentScoringStrategy = componentScoringStrategy;
        this.cryptaIdDispenser = cryptaIdDispenser;
        this.splitAlgorithm = splitAlgorithm;
    }
    public CalculateComponentInfoReducer(
            EdgeInfoProvider edgeInfoProvider,
            ComponentScoringStrategy componentScoringStrategy,
            CryptaIdDispenser cryptaIdDispenser,
            SplitAlgorithm splitAlgorithm,
            int recsLimit) {

        this(edgeInfoProvider, componentScoringStrategy, cryptaIdDispenser, splitAlgorithm);
        if (recsLimit > 0) {
            this.recsLimit = recsLimit;
        }
    }

    @Override
    public String key(YTreeMapNode entry) {
        return entry.getString(ComponentCenter.CRYPTA_ID_COLUMN);
    }

    private CollectionF<Component> getFinalComponents(Option<ComponentSplitDendrogram> split, Component component) {
        if (split.isPresent() && (split.get().getStatus().equals(MergeOfferStatus.SPLIT) ||
                split.get().getStatus().equals(MergeOfferStatus.SPLIT_BY_VERTEX))) {
            return split.get().allSplitComponents();
        } else {
            return Cf.list(component);
        }
    }

    private String yieldComponent(Component component, GraphInfo graphInfo, Yield<YTreeMapNode> yield, String origCryptaId) {
        componentScoringStrategy.setScoreTree(component, graphInfo);

        Option<ComponentCenter> cryptaIdO = cryptaIdDispenser.getCryptaId(component, graphInfo);
        if (cryptaIdO.isPresent()) {
            ComponentCenter componentCenter = cryptaIdO.get();
            String cryptaId = componentCenter.getCryptaId();
            componentCenter.setCcNeighboursCount(graphInfo.neighbourMergeOffers.size());

            component.setCryptaId(cryptaId);

            yield.yield(COMPONENT_OUT_INDEX, serialize(new GraphRecord(cryptaId, component, graphInfo)));

            MarkOutIndeviceEdges markOutIndeviceEdges = new MarkOutIndeviceEdges(
                    graphInfo,
                    component,
                    edgeInfoProvider);
            Stream<IndeviceLink> indeviceLinkStream = markOutIndeviceEdges.markOutIndeviceEdges();
            indeviceLinkStream.forEach(l -> yield.yield(INDEVICE_LINKS, serialize(l)));

            // vertices and edges
            yieldComponentInternals(yield, component);

            // vps
            for (Vertex vertex : component.getVertices()) {
                Option<YTreeMapNode> vpRec = graphInfo.verticesProperties.getRawRecs().getO(vertex);
                if (vpRec.isPresent()) {
                    YTreeMapNode node = vpRec.get();
                    node.put(ComponentCenter.CRYPTA_ID_COLUMN, YTree.stringNode(component.getCryptaId()));
                    yield.yield(VERTICES_PROPERTIES_OUT_INDEX, node);
                }
            }

            // stats
            ComponentStats stats = ComponentStats.calculate(componentCenter, component);
            yield.yield(COMPONENT_STATS_INDEX, serialize(stats));

            if (!cryptaId.equals(origCryptaId)) {
                yield.yield(CRYPTA_ID_CHANGE_STATS_INDEX, serialize(new CryptaIdChangedEvent(
                        origCryptaId, cryptaId
                )));
            }

            for (MergeOffer offer : graphInfo.neighbourMergeOffers) {
                if (!offer.getToCryptaId().equals(cryptaId)) {
                    // allows to track merge offers between components after crypta id change
                    yield.yield(NEIGHBOUR_MERGE_OFFERS_INDEX, serialize(
                            offer.copyWithChangedCryptaId(cryptaId)
                    ));
                }
            }
            return cryptaId;
        } else {
            yield.yield(STRANGE_INDEX, serialize(component));
        }
        return component.getCryptaId();
    }

    private SplitInfo.InnerSplitOffer getSplitOffer(ComponentSplitDendrogram componentSplitDendrogram) {
        String rightCryptaId = componentSplitDendrogram.getRightComponent().getCryptaId();
        String leftCryptaId = componentSplitDendrogram.getLeftComponent().getCryptaId();
        String cryptaId = componentSplitDendrogram.getCryptaId();

        Check.notNull(cryptaId);
        Check.notNull(rightCryptaId);
        Check.notNull(leftCryptaId);

        double edgeScore = this.edgeInfoProvider.getMultiEdgeScore(componentSplitDendrogram.getEdgesBetween()).getWeight();
        MetricsTree scoreGain = componentSplitDendrogram.getScoreGain();

        SplitInfo.InnerSplitOffer.Builder offerBuilder = SplitInfo.InnerSplitOffer.newBuilder()
                .setCryptaId(cryptaId)
                .setLeftCryptaId(leftCryptaId)
                .setRightCryptaId(rightCryptaId)
                .setEdgesScore(edgeScore)
                .setStatus(String.valueOf(componentSplitDendrogram.getStatus()))
                .setScore(scoreGain.getScore());

        SplitInfo.InnerSplitOffer.Scores.Builder scores = SplitInfo.InnerSplitOffer.Scores.newBuilder();
        for (Tuple2<String, Double> item : scoreGain.getChildren().entries()) {
            if (Math.abs(item.get2()) < Double.MIN_VALUE) {
                continue;
            }
            SplitInfo.InnerSplitOffer.Score.Builder score =
                    SplitInfo.InnerSplitOffer.Score
                            .newBuilder()
                            .setName(item.get1())
                            .setValue(item.get2());
            scores.addValues(score.build());
        }
        offerBuilder.setScores(scores.build());
        return offerBuilder.build();
    }

    private void yieldSplitInfo(ComponentSplitDendrogram split, CollectionF<Component> finalComponents,
            HashMap<Vertex, String> vertexToComponent, Yield<YTreeMapNode> yield) {

        SplitInfo.InnerSplitOffers.Builder offers = SplitInfo.InnerSplitOffers.newBuilder();
        ArrayList<Edge> edgesBetweenAfterSplitting = new ArrayList<>();

        for (ComponentSplitDendrogram componentSplitDendrogram : split.flattenTree()) {
            offers.addValues(getSplitOffer(componentSplitDendrogram));

            if (componentSplitDendrogram.getStatus().equals(MergeOfferStatus.SPLIT)) {
                // to yield edge_between (with neighbour of original component) with right cryptaId
                edgesBetweenAfterSplitting.addAll(componentSplitDendrogram.getEdgesBetween());
            }
        }

        byte[] bytesOffers = offers.build().toByteArray();
        // yield splitInfo with all finalComponent
        int splitInfoTableIndex = finalComponents.size() > 1 ? SPLIT_INFO_INDEX : SPLIT_INFO_NOT_SPLITTED_INDEX;
        for (Component component : finalComponents) {
            ru.yandex.crypta.graph2.model.matching.merge.SplitInfo splitInfo =
                    new ru.yandex.crypta.graph2.model.matching.merge.SplitInfo(component.getCryptaId(), bytesOffers);

            yield.yield(splitInfoTableIndex, serialize(splitInfo));
        }

        yieldEdgeBetweenAfterSplitting(edgesBetweenAfterSplitting, vertexToComponent, yield);
    }

    @Override
    public void reduce(String origComponentCenter, IteratorF<YTreeMapNode> entries, Yield<YTreeMapNode> yield,
                       Statistics statistics)
    {

        GraphInfoRecsParser parser = new GraphInfoRecsParser(this, OOM_INDEX, STRANGE_INDEX, recsLimit);
        Option<GraphInfo> graphInfoO = parser.parseInput(entries, yield);
        if (!graphInfoO.isPresent()) {
            return;
        }

        GraphInfo graphInfo = graphInfoO.get();

        CollectionF<Component> components = graphInfo.getComponents();
        Check.sizeIs(1, components);
        Component origComponent = components.iterator().next();

        ComponentConnectivityInspector connectivityInspector = new ComponentConnectivityInspector(origComponent);

        ListF<Component> origSubComponents = Cf.arrayList();
        if (connectivityInspector.isSingleComponent()) {
            origSubComponents.add(origComponent);
        } else {
            origSubComponents.addAll(connectivityInspector.splitToComponents());
        }

        HashMap<Vertex, String> vertexToComponent = new HashMap<>();
        HashMap<String, ComponentCenter> componentCentersByCryptaId = new HashMap<>();
        for (Component subComponent : origSubComponents) {
            Option<ComponentSplitDendrogram> maybeSplit = splitAlgorithm.split(subComponent, graphInfo);
            if (maybeSplit.isPresent()) {
                maybeSplit.get().setCryptaId(origComponentCenter);
                maybeSplit.get().setComponentCentersRecursively(cryptaIdDispenser, graphInfo, componentCentersByCryptaId);
            }
            CollectionF<Component> finalComponents = getFinalComponents(maybeSplit, subComponent);

            for (Component finalComponent : finalComponents) {
                String cryptaId = yieldComponent(finalComponent, graphInfo, yield, origComponentCenter);
                finalComponent.getVertices().forEach(v -> {vertexToComponent.put(v, cryptaId);});
            }

            if (maybeSplit.isPresent()) {
                yieldSplitInfo(maybeSplit.get(), finalComponents, vertexToComponent, yield);
            }
        }

        if (graphInfo.restRecords.isPresent()) {
            IteratorF<YTreeMapNode> edgesBetween = graphInfo.restRecords.get();
            while (edgesBetween.hasNext()) {
                EdgeBetweenWithNewCryptaIds edgeBetweenComponents =
                        parse(edgesBetween.next(), EdgeBetweenWithNewCryptaIds.class);
                Edge edge = edgeBetweenComponents.getEdge();
                MergeKeyWithNewCryptaIds mk = edgeBetweenComponents.getMergeKey();

                String edgeComponentLeft = vertexToComponent.getOrDefault(edge.getVertex1(), "");
                String edgeComponentRight = vertexToComponent.getOrDefault(edge.getVertex2(), "");
                if (edgeBetweenComponents.isOpposite()) {
                    String leftTmp = edgeComponentLeft;
                    edgeComponentLeft = edgeComponentRight;
                    edgeComponentRight = leftTmp;
                }
                if (!edgeComponentLeft.isEmpty()) {
                    mk.updateFromTo(
                            edgeBetweenComponents.getLeftCryptaId(),
                            edgeComponentLeft
                    );
                }
                if (!edgeComponentRight.isEmpty()) {
                    mk.updateFromTo(
                            edgeBetweenComponents.getRightCryptaId(),
                            edgeComponentRight
                    );
                }

                EdgeBetweenWithNewCryptaIds out =
                        new EdgeBetweenWithNewCryptaIds(edge, mk, edgeBetweenComponents.isOpposite());
                yield.yield(EDGES_BETWEEN, serialize(out));
            }
        }
    }

    private void yieldEdgeBetweenAfterSplitting(ArrayList<Edge> edgesBetweenAfterSplitting,
            HashMap<Vertex, String> vertexToComponent, Yield<YTreeMapNode> yield) {
        for (Edge edge : edgesBetweenAfterSplitting) {
            String leftCryptaId = vertexToComponent.get(edge.getVertex1());
            String rightCryptaId = vertexToComponent.get(edge.getVertex2());

            EdgeBetweenComponents edgeBetween = EdgeBetweenComponents.fromEdge(
                    edge,
                    leftCryptaId,
                    rightCryptaId
            );
            edgeBetween.setMergeKey(MergeKey.betweenComponents(leftCryptaId, rightCryptaId));

            yield.yield(EDGES_BETWEEN_SPLITTED_OUT_INDEX, serialize(edgeBetween));
        }
    }


    private void yieldComponentInternals(Yield<YTreeMapNode> yield, Component component) {

        String cryptaId = component.getCryptaId();
        // inner edges and vertices
        for (Vertex vertex : component.getVertices()) {
            VertexInComponent vertexInComponent = VertexInComponent.fromVertex(vertex, cryptaId);
            yield.yield(VERTICES_OUT_INDEX, serialize(vertexInComponent));
        }

        for (Edge edge : component.getInnerEdges()) {
            yield.yield(EDGES_OUT_INDEX, serialize(EdgeInComponent.fromEdge(edge, cryptaId)));
        }
    }
}
