package ru.yandex.crypta.graph2.matching.human.workflow.component.ops.indevice;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Queue;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jgrapht.Graph;
import org.jgrapht.Graphs;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.graph.soup.config.proto.ELogSourceType;
import ru.yandex.crypta.graph.soup.config.proto.ESourceType;
import ru.yandex.crypta.graph2.model.matching.component.Component;
import ru.yandex.crypta.graph2.model.matching.component.GraphInfo;
import ru.yandex.crypta.graph2.model.matching.graph.JGraphTUtils;
import ru.yandex.crypta.graph2.model.soup.edge.Edge;
import ru.yandex.crypta.graph2.model.soup.edge.weight.EdgeInfoProvider;
import ru.yandex.crypta.graph2.model.soup.props.DeviceId;
import ru.yandex.crypta.graph2.model.soup.props.UaProfile;
import ru.yandex.crypta.graph2.model.soup.props.UaProfile.UserAgentEquality;
import ru.yandex.crypta.graph2.model.soup.props.VertexPropertiesCollector;
import ru.yandex.crypta.graph2.model.soup.props.Yandexuid;
import ru.yandex.crypta.graph2.model.soup.vertex.Vertex;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;

import static ru.yandex.crypta.graph2.matching.human.workflow.component.ops.indevice.IndeviceLink.HEURISTIC_MATCH_TYPE;
import static ru.yandex.crypta.graph2.matching.human.workflow.component.ops.indevice.IndeviceLink.INDEVICE_MATCH_TYPE;

/**
 * For each device id it marks paths to yandexuids in two ways:
 * - for paths that suppose to by in-device by config, it checks this path by comparing ua_profiles
 * - for paths that suppose to by cross-device by config, tries to find potential in-device paths by comparing
 * ua_profiles
 */
public class MarkOutIndeviceEdges {

    static final int MAX_IN_DEVICE_DEPTH = 20;
    private final Component component;
    private final VertexPropertiesCollector verticesProperties;
    private final Graph<Vertex, Edge> graph;
    private EdgeInfoProvider edgeInfoProvider;

    public MarkOutIndeviceEdges(GraphInfo graphInfo,
                                Component component,
                                EdgeInfoProvider edgeInfoProvider) {
        this.component = component;
        this.verticesProperties = graphInfo.verticesProperties;
        this.graph = JGraphTUtils.toGraph(component.getInnerEdges(), component.getVertices());

        this.edgeInfoProvider = edgeInfoProvider;
    }

    public Stream<IndeviceLink> markOutIndeviceEdges() {
        List<IndeviceLink> result = new ArrayList<>();
        for (Vertex vertex : graph.vertexSet()) {
            if (Soup.CONFIG.isDeviceIdMainId(vertex.getIdType())) {
                result.addAll(markOutCurrentDeviceIndeviceEdges(vertex).collect(Collectors.toList()));
            }
        }
        return result.stream();
    }

    private Stream<IndeviceLink> markOutCurrentDeviceIndeviceEdges(Vertex deviceIdVertex) {
        DeviceId deviceId = verticesProperties.getDeviceIds().getTs(deviceIdVertex);
        if (deviceId == null) {
            return Stream.empty();
        }

        MarkOutIndeviceEdgesBFC markOutIndeviceEdgesBFC = new MarkOutIndeviceEdgesBFC(
                deviceIdVertex,
                MAX_IN_DEVICE_DEPTH
        );
        markOutIndeviceEdgesBFC.runAndMark();

        Set<VertexWithPath> strictIndevice = markOutIndeviceEdgesBFC.getTargetVertices();
        Set<Vertex> heuristicIndevice = addHeuristicIndeviceEdges(markOutIndeviceEdgesBFC, deviceIdVertex);

        return Stream.concat(
                strictIndevice.stream().map(v -> new IndeviceLink(
                        deviceIdVertex, v.getVertex(), v.getUserAgentCheck(), INDEVICE_MATCH_TYPE)
                ),
                heuristicIndevice.stream().map(v -> new IndeviceLink(
                        deviceIdVertex, v, UserAgentEquality.EQUALS, HEURISTIC_MATCH_TYPE
                ))
        ).peek(l -> {
            l.setDeviceIdUaProfile(deviceId.getUaProfile());
            l.setTargetIdUaProfile(verticesProperties.getYuids()
                    .getO(l.getTargetVertex())
                    .map(Yandexuid::getUaProfile)
                    .getOrElse(""));
        });
    }

    private Set<Vertex> addHeuristicIndeviceEdges(MarkOutIndeviceEdgesBFC markOutIndeviceEdgesBFC,
                                                  Vertex deviceIdVertex) {
        Set<Vertex> alreadyMarkedVertices = markOutIndeviceEdgesBFC.getTargetVertices()
                .stream()
                .map(VertexWithPath::getVertex)
                .collect(Collectors.toSet());

        // adds new heuristic indevice edges between device id and yuids that are not
        // reachable from device id by indevice edges
        Set<Vertex> heuristicIndeviceYandexuids = new HashSet<>();

        for (Vertex vertex : markOutIndeviceEdgesBFC.getVisitedVertices()) {
            if (vertex.getIdType().equals(EIdType.YANDEXUID) && !alreadyMarkedVertices.contains(vertex)) {
                if (isSameMobileDevice(deviceIdVertex, vertex) == UserAgentEquality.EQUALS) {
                    addHeuristicIndeviceEdge(deviceIdVertex, vertex);
                    heuristicIndeviceYandexuids.add(vertex);
                }
            }
        }

        return heuristicIndeviceYandexuids;
    }

    private void addHeuristicIndeviceEdge(Vertex deviceIdVertex, Vertex yuidVertex) {
        Edge newIndeviceEdge = new Edge(
                yuidVertex.getId(), yuidVertex.getIdType(),
                deviceIdVertex.getId(), deviceIdVertex.getIdType(),
                ESourceType.UA_MATCH, ELogSourceType.HEURISTIC,
                Cf.list(), Option.of(0.0), Option.of(0.0));
        newIndeviceEdge.setIndevice(true);
        component.addInnerEdge(newIndeviceEdge);
    }

    private UserAgentEquality isSameMobileDevice(Vertex deviceIdVertex, Vertex yuidVertex) {
        DeviceId deviceId = verticesProperties.getDeviceIds().getTs(deviceIdVertex);
        Yandexuid yuid = verticesProperties.getYuids().getTs(yuidVertex);
        if (deviceId == null || yuid == null) {
            return UserAgentEquality.NOT_EQUAL;
        }
        UaProfile deviceIdUaProfile = new UaProfile(deviceId.getUaProfile());
        UaProfile yuidUaProfile = new UaProfile(yuid.getUaProfile());
        return UaProfile.compareUaProfiles(deviceIdUaProfile, yuidUaProfile);
    }


    private class MarkOutIndeviceEdgesBFC {
        private final Vertex deviceIdVertex;
        private final Set<Vertex> visitedVertices = new HashSet<>();
        private final Queue<VertexWithPath> bfsQueue = new ArrayDeque<>();
        private final Set<VertexWithPath> targetVertices = new HashSet<>();
        private final int maxDepth;

        MarkOutIndeviceEdgesBFC(Vertex deviceIdVertex, int maxDepth) {
            this.deviceIdVertex = deviceIdVertex;
            this.maxDepth = maxDepth;
        }

        void runAndMark() {
            bfsQueue.add(new VertexWithPath(deviceIdVertex, new ArrayList<>()));
            visitedVertices.add(deviceIdVertex);

            while (!bfsQueue.isEmpty()) {
                VertexWithPath vertexPath = bfsQueue.poll();
                proceedCurrentVertex(vertexPath);

                Vertex vertex = vertexPath.getVertex();

                for (Edge edge : graph.edgesOf(vertex)) {
                    Vertex oppositeV = Graphs.getOppositeVertex(graph, edge, vertex);
                    if (!visitedVertices.contains(oppositeV) && vertexPath.getPathDepth() < maxDepth) {
                        bfsQueue.add(new VertexWithPath(vertexPath, edge, oppositeV));
                        visitedVertices.add(oppositeV);
                    }
                }

            }
        }


        private void proceedCurrentVertex(VertexWithPath currentEnqueuedVertex) {
            Vertex currentVertex = currentEnqueuedVertex.getVertex();
            if (currentVertex.getIdType().equals(EIdType.YANDEXUID) && currentEnqueuedVertex.isIndeviceByConfigPath()) {
                UserAgentEquality userAgentsCheck = isSameMobileDevice(deviceIdVertex, currentVertex);
                if (userAgentsCheck.atLeastRoughlyEquals()) {
                    targetVertices.add(currentEnqueuedVertex.withUserAgentCheck(userAgentsCheck));
                    markOutCurrentPathIndevice(currentEnqueuedVertex.getPathToVertex());
                }
            }
        }

        private void markOutCurrentPathIndevice(List<Edge> path) {
            for (Edge edge : path) {
                edge.setIndevice(true);
            }
        }

        Set<Vertex> getVisitedVertices() {
            return visitedVertices;
        }

        Set<VertexWithPath> getTargetVertices() {
            return targetVertices;
        }

    }

    private class VertexWithPath {
        private Vertex vertex;
        private List<Edge> pathToVertex = new ArrayList<>();
        private UserAgentEquality userAgentCheck;
        private boolean indeviceByConfigPath = true;

        VertexWithPath(Vertex vertex, List<Edge> pathToVertex) {
            this.vertex = vertex;
            this.pathToVertex.addAll(pathToVertex);
        }

        VertexWithPath(VertexWithPath prevEnqueuedVertex, Edge egdeToNextVertex, Vertex nextVertex) {
            this.vertex = nextVertex;
            this.pathToVertex.addAll(prevEnqueuedVertex.getPathToVertex());
            this.pathToVertex.add(egdeToNextVertex);

            boolean indeviceByConfigEdge = edgeInfoProvider.isIndevice(egdeToNextVertex);
            this.indeviceByConfigPath = prevEnqueuedVertex.indeviceByConfigPath && indeviceByConfigEdge;

        }

        VertexWithPath withUserAgentCheck(UserAgentEquality userAgentCheck) {
            this.userAgentCheck = userAgentCheck;
            return this;
        }

        Vertex getVertex() {
            return vertex;
        }

        List<Edge> getPathToVertex() {
            return pathToVertex;
        }

        int getPathDepth() {
            return pathToVertex.size();
        }

        UserAgentEquality getUserAgentCheck() {
            return userAgentCheck;
        }

        boolean isIndeviceByConfigPath() {
            return indeviceByConfigPath;
        }
    }
}
