package ru.yandex.crypta.graph2.matching.human.workflow.init.ops;

import java.util.List;
import java.util.Map;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.graph.soup.config.proto.TEdgeType;
import ru.yandex.crypta.graph2.dao.yt.proto.NativeProtobufOneOfMessageEntryType;
import ru.yandex.crypta.graph2.matching.human.proto.InitVertexCryptaIdsAndMergeNeighboursInRec;
import ru.yandex.crypta.graph2.matching.human.proto.InitVertexCryptaIdsAndMergeNeighboursOutRec;
import ru.yandex.crypta.graph2.model.id.proto.IdInfo;
import ru.yandex.crypta.graph2.model.matching.component.ComponentCenter;
import ru.yandex.crypta.graph2.model.matching.merge.MergeKeyType;
import ru.yandex.crypta.graph2.model.matching.proto.CryptaIdEdgeMessage;
import ru.yandex.crypta.graph2.model.matching.proto.EdgeProtoHelper;
import ru.yandex.crypta.graph2.model.matching.proto.MergeNeighbour;
import ru.yandex.crypta.graph2.model.matching.proto.VertexInComponent;
import ru.yandex.crypta.graph2.model.matching.proto.VertexKey;
import ru.yandex.crypta.graph2.model.matching.proto.VertexOverlimit;
import ru.yandex.crypta.graph2.model.soup.props.CommonShared;
import ru.yandex.crypta.graph2.model.soup.proto.EdgeType;
import ru.yandex.crypta.graph2.utils.IteratorUtils;
import ru.yandex.crypta.graph2.utils.IteratorUtils.IteratorSplit;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.inside.yt.kosher.impl.operations.utils.ReducerWithKey;
import ru.yandex.inside.yt.kosher.impl.operations.utils.YtSerializable;
import ru.yandex.inside.yt.kosher.operations.Statistics;
import ru.yandex.inside.yt.kosher.operations.Yield;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;

import static java.util.stream.Collectors.groupingBy;
import static ru.yandex.crypta.graph2.model.matching.proto.EdgeProtoHelper.BY_WEIGHT_DESC_COMPARATOR;

public class InitVertexCryptaIdsAndMergeNeighbours
        implements ReducerWithKey<InitVertexCryptaIdsAndMergeNeighboursInRec,
        InitVertexCryptaIdsAndMergeNeighboursOutRec,
        VertexKey>, YtSerializable {

    public static final String ID_COLUMN = "id";
    public static final String ID_TYPE_COLUMN = "id_type";
    public static final ListF<String> VERTEX_REDUCE_KEY = Cf.list(ID_COLUMN, ID_TYPE_COLUMN);
    private static final int OOM_LIMIT = 10000;

    private final List<TEdgeType> filterEdgeTypes;
    private final FilterSoupEdges filterSoupEdges;
    private final int edgesPerSourceLimit;

    public InitVertexCryptaIdsAndMergeNeighbours(int edgesPerSourceLimit) {
        this(edgesPerSourceLimit, List.of(), FilterSoupEdges.NONE);
    }

    public InitVertexCryptaIdsAndMergeNeighbours(int edgesPerSourceLimit,
                                                 List<TEdgeType> filterEdgeTypes,
                                                 FilterSoupEdges filterSoupEdges) {
        this.edgesPerSourceLimit = edgesPerSourceLimit;
        this.filterEdgeTypes = filterEdgeTypes;
        this.filterSoupEdges = filterSoupEdges;
    }

    private AllVertexMessages collectRecs(IteratorF<InitVertexCryptaIdsAndMergeNeighboursInRec> entries) {
        // all complexity of this method comes from the fact that we want correct recs overlimit handling

        IteratorSplit<InitVertexCryptaIdsAndMergeNeighboursInRec> table1RecsAndTail = IteratorUtils.takeWhile(
                entries,
                InitVertexCryptaIdsAndMergeNeighboursInRec::hasPrevCryptaIdVertexRec1
        );
        ListF<VertexInComponent> vertexPrevAssignment = table1RecsAndTail.getHead().map(
                InitVertexCryptaIdsAndMergeNeighboursInRec::getPrevCryptaIdVertexRec1
        );


        IteratorSplit<InitVertexCryptaIdsAndMergeNeighboursInRec> table2RecsAndTail = IteratorUtils.takeWhile(
                table1RecsAndTail.getTail(),
                InitVertexCryptaIdsAndMergeNeighboursInRec::hasIdInfoRec2
        );
        ListF<IdInfo> idsInfo = table2RecsAndTail.getHead().map(
                InitVertexCryptaIdsAndMergeNeighboursInRec::getIdInfoRec2
        );

        boolean isShared = idsInfo.find(
                info -> info.getSource().equals(CommonShared.COMMON_SHARED_SOURCE)
        ).isPresent();

        Tuple2<IteratorF<InitVertexCryptaIdsAndMergeNeighboursInRec>, Boolean> restRecs = IteratorUtils.checkOverlimit(
                table2RecsAndTail.getTail(), OOM_LIMIT
        );

        IteratorF<CryptaIdEdgeMessage> edgeMessagesIter = restRecs._1.map(
                InitVertexCryptaIdsAndMergeNeighboursInRec::getEdgeMessageRec3
        ).filter(this::filterEdge);
        Boolean isOverlimit = restRecs._2;

        return new AllVertexMessages(
                vertexPrevAssignment,
                idsInfo,
                edgeMessagesIter,
                isOverlimit,
                isShared
        );

    }

    private boolean filterEdge(CryptaIdEdgeMessage e) {
        if (filterSoupEdges == FilterSoupEdges.NONE) {
            return true;
        }

        TEdgeType edgeType = Soup.CONFIG.tryGetEdgeType(
                e.getId1Type(),
                e.getId2Type(),
                e.getSourceType(),
                e.getLogSource()
        );

        if (edgeType == null) {
            return false;
        }

        if (filterEdgeTypes.contains(edgeType)) {
            return false;
        }

        if (filterSoupEdges == FilterSoupEdges.PROD) {
            return Soup.CONFIG.getEdgeUsage(edgeType).getHumanMatching();
        } else if (filterSoupEdges == FilterSoupEdges.EXP) {
            return Soup.CONFIG.getEdgeUsage(edgeType).getHumanMatchingExp();
        }

        return true;
    }

    public static class AllVertexMessages {
        public final ListF<VertexInComponent> prevStepAssignment;
        public final ListF<IdInfo> idInfoRecs;
        public final IteratorF<CryptaIdEdgeMessage> edgeMessages;
        public final boolean starOverlimit;
        public final boolean shared;

        public AllVertexMessages(ListF<VertexInComponent> prevStepAssignment,
                                 ListF<IdInfo> idInfoRecs,
                                 IteratorF<CryptaIdEdgeMessage> edgeMessages,
                                 boolean starOverlimit, boolean shared) {
            this.prevStepAssignment = prevStepAssignment;
            this.idInfoRecs = idInfoRecs;
            this.edgeMessages = edgeMessages;
            this.starOverlimit = starOverlimit;
            this.shared = shared;
        }
    }

    @Override
    public VertexKey key(InitVertexCryptaIdsAndMergeNeighboursInRec entry) {

        VertexKey.Builder builder = VertexKey.newBuilder();

        if (entry.hasPrevCryptaIdVertexRec1()) {
            VertexInComponent table1 = entry.getPrevCryptaIdVertexRec1();
            return builder.setId(table1.getId()).setIdType(table1.getIdType()).build();
        } else if (entry.hasIdInfoRec2()) {
            IdInfo table2 = entry.getIdInfoRec2();
            return builder.setId(table2.getId()).setIdType(table2.getIdType()).build();
        } else if (entry.hasEdgeMessageRec3()) {
            CryptaIdEdgeMessage table4 = entry.getEdgeMessageRec3();
            return builder.setId(table4.getId()).setIdType(table4.getIdType()).build();
        } else {
            throw new IllegalArgumentException("Rec type is not supported " + entry);
        }

    }


    @Override
    public void reduce(VertexKey sourceVertex, IteratorF<InitVertexCryptaIdsAndMergeNeighboursInRec> entries,
                       Yield<InitVertexCryptaIdsAndMergeNeighboursOutRec> yield,
                       Statistics statistics) {
        AllVertexMessages parsed = collectRecs(entries);
        IteratorF<CryptaIdEdgeMessage> blankMessages = parsed.edgeMessages;
        if (parsed.shared) {
            blankMessages
                    .forEachRemaining(blankMessage ->
                            yield.yield(
                                    InitVertexCryptaIdsAndMergeNeighboursOutRec.newBuilder()
                                            .setSharedEdgeRec7(blankMessage)
                                            .build())
                    );
            return;
        }

        if (blankMessages.hasNext()) {

            ListF<String> vertexCryptaIds;

            if (parsed.prevStepAssignment.isNotEmpty()) {
                SetF<String> prevStepCryptaIds = parsed.prevStepAssignment
                        .map(VertexInComponent::getCryptaId)
                        .unique();
                boolean isMultiComponent = prevStepCryptaIds.size() > 1;

                // current vertex already connects several components
                for (String cryptaId : prevStepCryptaIds) {
                    VertexInComponent out = VertexInComponent.newBuilder()
                            .setId(sourceVertex.getId())
                            .setIdType(sourceVertex.getIdType())
                            .setCryptaId(cryptaId)
                            .build();

                    yield.yield(InitVertexCryptaIdsAndMergeNeighboursOutRec.newBuilder()
                            .setCryptaIdVertexRec1(out)
                            .build()
                    );

                    if (isMultiComponent) {
                        MergeNeighbour mergeNeighbour = MergeNeighbour.newBuilder()
                                .setCryptaId(cryptaId)
                                .setMergeKey(sourceVertex.getId())
                                .setMergeKeyType(MergeKeyType.BY_VERTEX.name())
                                .build();

                        yield.yield(InitVertexCryptaIdsAndMergeNeighboursOutRec.newBuilder()
                                .setMergeNeighbourRec4(mergeNeighbour)
                                .build()
                        );
                    }
                }

                vertexCryptaIds = prevStepCryptaIds.toList();

            } else {
                // new vertex in soup
                EIdType idType = Soup.CONFIG.getIdType(sourceVertex.getIdType()).getType();
                String newCryptaId = ComponentCenter.toCryptaId(sourceVertex.getId(), idType);

                VertexInComponent out = VertexInComponent.newBuilder()
                        .setId(sourceVertex.getId())
                        .setIdType(sourceVertex.getIdType())
                        .setCryptaId(newCryptaId)
                        .build();

                yield.yield(InitVertexCryptaIdsAndMergeNeighboursOutRec.newBuilder()
                        .setCryptaIdVertexRec1(out)
                        .build()
                );

                vertexCryptaIds = Cf.list(newCryptaId);
            }

            if (!parsed.starOverlimit) {

                // in fact we would like to track all edge messages even in case of overlimit
                // but for huge stars it affects performance too much
                // missing edge messages will be checked later here:
                // ru.yandex.crypta.graph2.matching.human.init.ops.InitEdgesCryptaIdsAndMergeNeighbours#reduce

                Map<EdgeType, List<CryptaIdEdgeMessage>> bySource = blankMessages.stream().collect(
                        groupingBy(EdgeProtoHelper::getEdgeType)
                );

                for (EdgeType edgeType : bySource.keySet()) {
                    // mark edge messages with crypta id
                    List<CryptaIdEdgeMessage> messages = bySource.get(edgeType);

                    // take actual edges first
                    messages.stream()
                            .sorted(BY_WEIGHT_DESC_COMPARATOR)
                            .limit(edgesPerSourceLimit)
                            .forEach(blankMessage -> {
                                for (String cryptaId : vertexCryptaIds) {
                                    // sending messages to detect crypta ids, joined by edges

                                    CryptaIdEdgeMessage messageWithOppositeRecipient =
                                            EdgeProtoHelper.reverse(blankMessage)
                                                    .setCryptaId(cryptaId)
                                                    .build();

                                    yield.yield(InitVertexCryptaIdsAndMergeNeighboursOutRec
                                            .newBuilder()
                                            .setCryptaIdMessageRec3(messageWithOppositeRecipient)
                                            .build()
                                    );
                                }
                            });

                    if (messages.size() > edgesPerSourceLimit) {
                        VertexOverlimit vertexOverlimit = VertexOverlimit.newBuilder()
                                .setId(sourceVertex.getId())
                                .setIdType(sourceVertex.getIdType())
                                .setLimit(edgesPerSourceLimit)
                                .setSource(edgeType.toString())
                                .build();

                        yield.yield(InitVertexCryptaIdsAndMergeNeighboursOutRec
                                .newBuilder()
                                .setOverlimitVertexRec5(vertexOverlimit)
                                .build()
                        );
                    }

                }

                // mark vertices properties with crypta id
                for (String cryptaId : vertexCryptaIds) {
                    for (IdInfo vpNode : parsed.idInfoRecs) {

                        IdInfo vpRecCopy = vpNode.toBuilder().setCryptaId(cryptaId).build();

                        yield.yield(InitVertexCryptaIdsAndMergeNeighboursOutRec
                                .newBuilder()
                                .setIdInfoRec2(vpRecCopy)
                                .build()
                        );
                    }
                }

            } else {
                VertexOverlimit vertexOverlimit = VertexOverlimit.newBuilder()
                        .setId(sourceVertex.getId())
                        .setIdType(sourceVertex.getIdType())
                        .setLimit(OOM_LIMIT)
                        .build();
                yield.yield(InitVertexCryptaIdsAndMergeNeighboursOutRec
                        .newBuilder()
                        .setOverlimitVertexRec5(vertexOverlimit)
                        .build()
                );
            }
        } else {
            if (parsed.prevStepAssignment.isNotEmpty()) {
                yield.yield(InitVertexCryptaIdsAndMergeNeighboursOutRec
                        .newBuilder()
                        .setOutdatedIndexRec6(sourceVertex)
                        .build()
                );
            }
            // otherwise it's vertex property without any matching
        }

    }

    @Override
    public YTableEntryType<InitVertexCryptaIdsAndMergeNeighboursInRec> inputType() {
        return new NativeProtobufOneOfMessageEntryType<>(
                InitVertexCryptaIdsAndMergeNeighboursInRec.newBuilder(), true
        );
    }

    @Override
    public YTableEntryType<InitVertexCryptaIdsAndMergeNeighboursOutRec> outputType() {

        return new NativeProtobufOneOfMessageEntryType<>(
                InitVertexCryptaIdsAndMergeNeighboursOutRec.newBuilder(), true
        );
    }

    public enum FilterSoupEdges {
        PROD,
        EXP,
        NONE
    }
}
