package ru.yandex.crypta.graph2.matching.human.workflow.merge_apply;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.matching.human.paths.ComponentsRawTables;
import ru.yandex.crypta.graph2.matching.human.paths.ComponentsRawTablesWithEdgesBetween;
import ru.yandex.crypta.graph2.matching.human.strategy.HumanMatchingStrategyProvider;
import ru.yandex.crypta.graph2.matching.human.workflow.HumanMatchingTask;
import ru.yandex.crypta.graph2.matching.human.workflow.merge_apply.ApplyMergeTask.ReadyForMergeTables;
import ru.yandex.crypta.graph2.matching.human.workflow.merge_apply.ops.AddChangedCryptaIdReducer;
import ru.yandex.crypta.graph2.matching.human.workflow.merge_apply.ops.ApplyMergeDecisionReducer;
import ru.yandex.crypta.graph2.matching.human.workflow.merge_apply.ops.ApplyMergeDecisionToEdgesBetweenReducer;
import ru.yandex.crypta.graph2.model.matching.component.ComponentCenter;
import ru.yandex.crypta.graph2.model.matching.edge.EdgeBetweenWithNewCryptaIds;
import ru.yandex.crypta.graph2.model.matching.edge.EdgeInComponent;
import ru.yandex.crypta.graph2.model.matching.merge.MergeKey;
import ru.yandex.crypta.graph2.model.matching.merge.MergeKeyWithNewCryptaIds;
import ru.yandex.crypta.graph2.model.matching.vertex.VertexInComponent;
import ru.yandex.crypta.graph2.model.soup.props.VertexPropertiesSchemaHelper;
import ru.yandex.crypta.graph2.workflow.IterableTables;
import ru.yandex.inside.yt.kosher.cypress.YPath;

public class ApplyMergeTask extends HumanMatchingTask<ReadyForMergeTables, ComponentsRawTablesWithEdgesBetween> {

    private static final Logger LOG = LoggerFactory.getLogger(ApplyMergeTask.class);

    private YPath mergedVertices;
    private YPath mergedVerticesProperties;
    private YPath mergedEdgesTmp1;
    private YPath mergedEdgesTmp2;
    private YPath mergedEdges;
    private YPath mergedBetweenEdges;
    private YPath mergeKeysWithNewCryptaIds;


    public ApplyMergeTask(Dao dao, YPath workdir, HumanMatchingStrategyProvider matchingStrategy, String nameSuffix) {
        super(dao, workdir, matchingStrategy, nameSuffix);

        mergedVertices = workdir.child("vertices_merged");
        mergedVerticesProperties = workdir.child("vertices_properties_merged");
        mergedEdgesTmp1 = workdir.child("edges_merged_tmp1");
        mergedEdgesTmp2 = workdir.child("edges_merged_tmp2");
        mergedEdges = workdir.child("edges_merged");

        mergedBetweenEdges = workdir.child("edges_between_unmerged");
        mergeKeysWithNewCryptaIds = workdir.child("merge_keys_with_new_cryptaids");
    }

    @Override
    protected void runImpl(ReadyForMergeTables readyForMergeTables) {

        dao.ytCypress().ensureDir(workdir);

        // we would prefer to apply merge offers by merge key,
        // but will do it by ccId,ccIdType to avoid using large merge key tables
        ListF<YPath> inTables = Cf.arrayList(
                readyForMergeTables.mergeOffers,
                readyForMergeTables.tablesToMerge.allComponentTables.verticesTable,
                readyForMergeTables.tablesToMerge.allComponentTables.verticesPropertiesTable,
                readyForMergeTables.tablesToMerge.allComponentTables.edgesTable
        );

        LOG.info("Preparing for for ApplyMergeDecisionReducer");
        dao.ytOps().sortAllParallel(inTables, ComponentCenter.CRYPTA_ID_KEY);

        dao.ytCypress().createTableWithSchema(mergedVertices, VertexInComponent.class);
        dao.ytCypress().createTableWithSchema(
                mergedVerticesProperties,
                VertexPropertiesSchemaHelper.getUnionVertexPropertiesSchema()
        );
        dao.ytCypress().createTableWithSchema(mergedEdgesTmp1, EdgeInComponent.class);

        LOG.info("Running ApplyMergeDecisionReducer");
        dao.ytOps().reduceSync(
                inTables,
                Cf.list(
                        mergedVertices,
                        mergedVerticesProperties,
                        mergedEdgesTmp1
                ),
                ComponentCenter.CRYPTA_ID_KEY,
                new ApplyMergeDecisionReducer()
        );

        applyMergeDecisionsToEdgesBetween(
                readyForMergeTables.mergeOffers,
                readyForMergeTables.mergeNeighbours,
                readyForMergeTables.tablesToMerge.edgesBetweenComponents,
                mergedEdgesTmp2,
                mergedBetweenEdges
        );

        // concat two types of edges
        ListF<YPath> edgesTmp = Cf.list(
                mergedEdgesTmp1,
                mergedEdgesTmp2
        );
        dao.ytCypress().createTableWithSchema(mergedEdges, EdgeInComponent.class);
        dao.yt().cypress().concatenate(edgesTmp, mergedEdges);
        dao.ytCypress().removeAll(edgesTmp);
    }

    @Override
    public ComponentsRawTablesWithEdgesBetween getOutput() {
        return new ComponentsRawTablesWithEdgesBetween(
                new ComponentsRawTables(
                        mergedVertices,
                        mergedVerticesProperties,
                        mergedEdges
                ),
                mergedBetweenEdges
        );
    }

    public void applyMergeDecisionsToEdgesBetween(YPath decisions, YPath mergeNeighbours, YPath edgesBetween, YPath edgesInComponentResult, YPath edgesBetweenComponentsResult) {
        LOG.info("Apply merge decisions to edges between components");

        var inputs = Cf.list(decisions, mergeNeighbours);
        dao.ytOps().sortAllParallel(inputs, ComponentCenter.CRYPTA_ID_KEY);

        dao.ytCypress().createTableWithSchema(mergeKeysWithNewCryptaIds, MergeKeyWithNewCryptaIds.class);

        LOG.info("Running AddChangedCryptaIdReducer");
        dao.ytOps().reduceSync(
                inputs,
                Cf.list(
                        mergeKeysWithNewCryptaIds
                ),
                ComponentCenter.CRYPTA_ID_KEY,
                new AddChangedCryptaIdReducer()
        );

        dao.ytOps().sortSync(mergeKeysWithNewCryptaIds, MergeKey.REDUCE_KEY);

        dao.ytCypress().createTableWithSchema(edgesInComponentResult, EdgeInComponent.class);
        dao.ytCypress().createTableWithSchema(edgesBetweenComponentsResult, EdgeBetweenWithNewCryptaIds.class);

        LOG.info("Running ApplyMergeDecisionToEdgesBetweenReducer");
        dao.ytOps().reduceSync(
                Cf.list(mergeKeysWithNewCryptaIds, edgesBetween),
                Cf.list(
                        edgesInComponentResult,
                        edgesBetweenComponentsResult
                ),
                MergeKey.REDUCE_KEY,
                new ApplyMergeDecisionToEdgesBetweenReducer()
        );
    }

    @Override
    public String getDescription() {
        return "Merge components by confirmed offers";
    }

    public static class ReadyForMergeTables extends IterableTables {
        public ComponentsRawTablesWithEdgesBetween tablesToMerge;
        public YPath mergeNeighbours;
        public YPath mergeOffers;

        public ReadyForMergeTables(ComponentsRawTablesWithEdgesBetween tablesToMerge, YPath mergeNeighbours, YPath mergeOffers) {
            this.tablesToMerge = tablesToMerge;
            this.mergeNeighbours = mergeNeighbours;
            this.mergeOffers = mergeOffers;
        }

        @Override
        public ListF<YPath> allTables() {
            return Cf.list(mergeNeighbours, mergeOffers).plus(tablesToMerge.allTables());
        }
    }

}
