package ru.yandex.crypta.graph2.matching.human.workflow.neighbours;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.matching.human.strategy.HumanMatchingStrategyProvider;
import ru.yandex.crypta.graph2.matching.human.workflow.HumanMatchingTask;
import ru.yandex.crypta.graph2.matching.human.workflow.component.ProcessComponentsTask;
import ru.yandex.crypta.graph2.matching.human.workflow.neighbours.PostProcessMergeOffers.NeighboursInfo;
import ru.yandex.crypta.graph2.matching.human.workflow.neighbours.ops.CalculateComponentNeighboursReducer;
import ru.yandex.crypta.graph2.matching.human.workflow.neighbours.ops.ConvertEdgeBetweenToEdgeInMapper;
import ru.yandex.crypta.graph2.matching.human.workflow.prepare.ConcatTablesHelper;
import ru.yandex.crypta.graph2.model.matching.component.ComponentCenter;
import ru.yandex.crypta.graph2.model.matching.component.ComponentNeighbours;
import ru.yandex.crypta.graph2.model.matching.edge.EdgeBetweenComponents;
import ru.yandex.crypta.graph2.model.matching.edge.EdgeInComponent;
import ru.yandex.crypta.graph2.model.matching.merge.MergeKey;
import ru.yandex.crypta.graph2.workflow.IterableTables;
import ru.yandex.inside.yt.kosher.cypress.YPath;

public class PostProcessMergeOffers extends HumanMatchingTask<ProcessComponentsTask.ProcessComponentTables, NeighboursInfo> {

    private static final Logger LOG = LoggerFactory.getLogger(PostProcessMergeOffers.class);

    private YPath neighbours;
    private YPath edgesBetweenByCryptaId;
    private YPath edgesBetweenTmp;

    public PostProcessMergeOffers(Dao dao, YPath workdir, HumanMatchingStrategyProvider matchingStrategy, String nameSuffix) {
        super(dao, workdir, matchingStrategy, nameSuffix);

        neighbours = workdir.child("neighbours");
        edgesBetweenByCryptaId = workdir.child("edges_between_final");
        edgesBetweenTmp = workdir.child("edges_between_tmp");
    }

    private MergeResults prepareForMergeOffersProcessing(
            ProcessComponentsTask.ProcessComponentTables componentsProcessed) {
        dao.ytCypress().ensureDir(workdir);

        new ConcatTablesHelper(dao).mergeTablesWithSchemaFromOutput(
                Cf.list(
                        componentsProcessed.edgesBetween,  // after merge
                        componentsProcessed.graphTables.edgesBetweenComponents  // after split
                ),
                edgesBetweenTmp,
                EdgeBetweenComponents.class
        );

        return new MergeResults(
                componentsProcessed.mergeOffers, edgesBetweenTmp
        );
    }

    @Override
    protected void runImpl(ProcessComponentsTask.ProcessComponentTables componentsProcessed) {
        dao.ytCypress().ensureDir(workdir);

        MergeResults in = prepareForMergeOffersProcessing(componentsProcessed);

        LOG.info("Preparing for for ProcessMergeOffers");
        dao.ytOps().sortAllParallel(Cf.list(in.mergeOffers, in.edgesBetween), MergeKey.REDUCE_KEY);
        dao.ytCypress().createTableWithSchema(edgesBetweenByCryptaId, EdgeInComponent.class);
        dao.ytCypress().createTableWithSchema(neighbours, ComponentNeighbours.class);

        dao.ytOps().mapSync(
                Cf.list(in.edgesBetween),
                Cf.list(edgesBetweenByCryptaId),
                new ConvertEdgeBetweenToEdgeInMapper()
        );
        dao.ytOps().sortSync(edgesBetweenByCryptaId, ComponentCenter.CRYPTA_ID_KEY);
        dao.ytOps().reduceSync(
                Cf.list(edgesBetweenByCryptaId),
                Cf.list(neighbours),
                ComponentCenter.CRYPTA_ID_KEY,
                new CalculateComponentNeighboursReducer()
        );
    }

    @Override
    public NeighboursInfo getOutput() {
        return new NeighboursInfo(
                neighbours,
                edgesBetweenByCryptaId,
                edgesBetweenTmp
        );
    }

    @Override
    public String getDescription() {
        return "Tries to collect neighbour crypta ids from strange sources";
    }

    public static class MergeResults {
        public YPath mergeOffers;
        public YPath edgesBetween;

        public MergeResults(YPath mergeOffers, YPath edgesBetween) {
            this.mergeOffers = mergeOffers;
            this.edgesBetween = edgesBetween;
        }
    }

    public static class NeighboursInfo extends IterableTables {
        public YPath neighbours;
        public YPath edgesBetweenByCryptaId;
        public YPath edgesBetween;

        public NeighboursInfo(YPath neighbours, YPath edgesBetweenByCryptaId, YPath edgesBetweeen) {
            this.neighbours = neighbours;
            this.edgesBetweenByCryptaId = edgesBetweenByCryptaId;
            this.edgesBetween = edgesBetweeen;
        }

        @Override
        public ListF<YPath> allTables() {
            return Cf.list(neighbours, edgesBetweenByCryptaId, edgesBetween);
        }
    }
}
