package ru.yandex.crypta.graph2.matching.human.workflow.output;

import java.time.LocalDate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.dao.yt.TableWithTmp;
import ru.yandex.crypta.graph2.dao.yt.ops.Await;
import ru.yandex.crypta.graph2.matching.human.helper.YQLTypeConfigGenerator;
import ru.yandex.crypta.graph2.matching.human.paths.ComponentsRawTables;
import ru.yandex.crypta.graph2.matching.human.strategy.HumanMatchingStrategyProvider;
import ru.yandex.crypta.graph2.matching.human.workflow.neighbours.PostProcessMergeOffers.NeighboursInfo;
import ru.yandex.crypta.graph2.matching.human.workflow.output.CopyToOutputTask.CopyToOutputTables;
import ru.yandex.crypta.graph2.model.matching.component.ComponentCenter;
import ru.yandex.crypta.graph2.model.matching.edge.EdgeInComponent;
import ru.yandex.crypta.graph2.model.soup.edge.Edge;
import ru.yandex.crypta.graph2.model.soup.vertex.Vertex;
import ru.yandex.crypta.graph2.workflow.Task;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.YPath;

public class CopyToOutputTask extends Task<CopyToOutputTables, YPath, HumanMatchingStrategyProvider> {

    private static final Logger LOG = LoggerFactory.getLogger(CopyToOutputTask.class);
    private final LocalDate soupGenerationDate;

    public CopyToOutputTask(Dao dao, YPath workdir, HumanMatchingStrategyProvider matchingStrategy,
                            LocalDate soupGenerationDate, String nameSuffix) {
        super(dao, workdir, matchingStrategy, nameSuffix);
        this.soupGenerationDate = soupGenerationDate;
    }

    @Override
    public YPath getOutput() {
        return workdir;
    }

    @Override
    public String getDescription() {
        return "Copies tables to dicts with proper sorting";
    }

    @Override
    protected void runImpl(CopyToOutputTables input) {
        dao.ytCypress().ensureDir(workdir);

        LOG.info("Last sorting copy");

        dao.ytTr().withTransactionContext(dao, txContext -> {
            Option<GUID> txId = Option.of(txContext.getTransactionId());

            ListF<CopyConfig> copy = Cf.arrayList();
            copy.add(new CopyConfig(
                    input.bestMatchTables.verticesTable,
                    txContext.createNoSchema(workdir.child("vertices_no_multi_profile")),
                    Vertex.VERTEX_REDUCE_KEY
            ));
            copy.add(new CopyConfig(
                    input.bestMatchTables.verticesTable,
                    txContext.createNoSchema(workdir.child("vertices_no_multi_profile_by_id_type")),
                    Vertex.ID_TYPE_FIRST_KEY
            ));
            copy.add(new CopyConfig(
                    input.bestMatchTables.verticesTable,
                    txContext.createNoSchema(workdir.child("vertices_no_multi_profile_by_crypta_id")),
                    ComponentCenter.CRYPTA_ID_KEY
            ));
            copy.add(new CopyConfig(
                    input.bestMatchTables.verticesPropertiesTable,
                    txContext.createNoSchema(workdir.child("vertices_properties_by_crypta_id")),
                    ComponentCenter.CRYPTA_ID_KEY
            ));
            copy.add(new CopyConfig(
                    input.bestMatchTables.edgesTable,
                    txContext.createNoSchema(workdir.child("edges_by_crypta_id")),
                    EdgeInComponent.EDGE_WITH_MULTI_CRYPTA_ID_KEY
            ));
            copy.add(new CopyConfig(
                    input.componentsStatsTable,
                    txContext.createNoSchema(workdir.child("crypta_components_stats")),
                    ComponentCenter.CRYPTA_ID_KEY
            ));
            copy.add(new CopyConfig(
                    input.mergeOffers,
                    txContext.createNoSchema(workdir.child("merge_offers")),
                    ComponentCenter.CRYPTA_ID_KEY
            ));
            copy.add(new CopyConfig(
                    input.splitInfo,
                    txContext.createNoSchema(workdir.child("split_info")),
                    ComponentCenter.CRYPTA_ID_KEY
            ));
            copy.add(new CopyConfig(
                    input.neighboursInfo.edgesBetweenByCryptaId,
                    txContext.createNoSchema(workdir.child("edges_between_components_by_crypta_id")),
                    EdgeInComponent.EDGE_WITH_MULTI_CRYPTA_ID_KEY
            ));
            copy.add(new CopyConfig(
                    input.neighboursInfo.edgesBetween,
                    txContext.createNoSchema(workdir.child("edges_between_components")),
                    Edge.EDGE_UNIQUE_KEY
            ));
            copy.add(new CopyConfig(
                    input.neighboursInfo.neighbours,
                    txContext.createNoSchema(workdir.child("crypta_components_neighbours")),
                    ComponentCenter.CRYPTA_ID_KEY
            ));
            copy.add(new CopyConfig(
                    input.components,
                    txContext.createNoSchema(workdir.child("components_by_crypta_id")),
                    ComponentCenter.CRYPTA_ID_KEY
            ));

            Await.all(
                    copy.map(copyConfig -> dao.ytOps().sortAsync(
                            txId,
                            Cf.list(copyConfig.from),
                            copyConfig.to.getTmpTablePath(),
                            copyConfig.sortBy
                    ))
            );

            txContext.finalizeTables(copy.map(x -> x.to), false);

            LocalDate endOfProcessingDate = LocalDate.now();
            copy.forEach(cp -> {
                dao.ytCypress().setRunDate(txId, cp.to.getTablePath(), endOfProcessingDate);
                dao.ytCypress().setGenerationDate(txId, cp.to.getTablePath(), soupGenerationDate);
            });

            setYQLProtoFields(txId, workdir.child("split_info"), workdir.child("components_by_crypta_id"));
        });

        LOG.info("Copy is done");
    }

    private void setYQLProtoFields(Option<GUID> txId, YPath splitInfo, YPath components) {
        try {
            dao.ytCypress().setAttribute(txId, splitInfo, "_yql_proto_field_offers", YQLTypeConfigGenerator.getYQlProtoFieldForSplitInfo());
            dao.ytCypress().setAttribute(txId, components, "_yql_proto_field_graph", YQLTypeConfigGenerator.getYQlProtoFieldForGraphEngineRecord());
        } catch (java.lang.UnsatisfiedLinkError e) {
            LOG.info("Cannot find matching-model_java");
        }
    }

    public static class CopyToOutputTables {
        public ComponentsRawTables bestMatchTables;
        public YPath componentsStatsTable;
        public NeighboursInfo neighboursInfo;
        public YPath mergeOffers;
        public YPath splitInfo;
        public YPath components;

        public CopyToOutputTables(ComponentsRawTables bestMatchTables, YPath componentsStatsTable,
                NeighboursInfo neighboursInfo, YPath mergeOffers, YPath splitInfo, YPath components) {
            this.bestMatchTables = bestMatchTables;
            this.componentsStatsTable = componentsStatsTable;
            this.neighboursInfo = neighboursInfo;
            this.mergeOffers = mergeOffers;
            this.splitInfo = splitInfo;
            this.components = components;
        }
    }

    private static class CopyConfig {
        private YPath from;
        private TableWithTmp to;
        private ListF<String> sortBy;

        public CopyConfig(YPath from, TableWithTmp to, ListF<String> sortBy) {
            this.from = from;
            this.to = to;
            this.sortBy = sortBy;
        }
    }


}
