package ru.yandex.crypta.graph2.matching.human.workflow.prepare;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.graph2.dao.Dao;
import ru.yandex.crypta.graph2.dao.yt.TableWithTmp;
import ru.yandex.crypta.graph2.matching.human.helper.YQLTypeConfigGenerator;
import ru.yandex.crypta.graph2.matching.human.paths.ComponentsRawTables;
import ru.yandex.crypta.graph2.matching.human.strategy.HumanMatchingStrategyProvider;
import ru.yandex.crypta.graph2.matching.human.workflow.HumanMatchingTask;
import ru.yandex.crypta.graph2.matching.human.workflow.init.InitComponentsAndNeighboursTask.GraphInitTables;
import ru.yandex.crypta.graph2.matching.human.workflow.prepare.PrepareComponentsForMergeTask.ComponentsByMergeKeyTables;
import ru.yandex.crypta.graph2.matching.human.workflow.prepare.ops.PutComponentsToMergeKey;
import ru.yandex.crypta.graph2.model.matching.component.ComponentCenter;
import ru.yandex.crypta.graph2.model.matching.edge.EdgeInComponent;
import ru.yandex.crypta.graph2.model.matching.proto.ComponentToMerge;
import ru.yandex.crypta.graph2.model.matching.vertex.VertexInComponent;
import ru.yandex.crypta.graph2.model.soup.props.VertexPropertiesSchemaHelper;
import ru.yandex.crypta.graph2.workflow.IterableTables;
import ru.yandex.inside.yt.kosher.cypress.YPath;

public class PrepareComponentsForMergeTask
        extends HumanMatchingTask<GraphInitTables, ComponentsByMergeKeyTables>
{

    private static final Logger LOG = LoggerFactory.getLogger(PrepareComponentsForMergeTask.class);

    private YPath componentsToMerge;

    private YPath verticesWithNeighbours;
    private YPath verticesPropertiesWithNeighbours;
    private YPath edgesWithNeighbours;

    private YPath verticesNoNeighbours;
    private YPath verticesPropertiesNoNeighbours;
    private YPath edgesNoNeighbours;

    public PrepareComponentsForMergeTask(Dao dao, YPath workdir, HumanMatchingStrategyProvider matchingStrategy, String nameSuffix) {
        super(dao, workdir, matchingStrategy, nameSuffix);

        componentsToMerge = workdir.child("components_by_merge_key");

        verticesWithNeighbours = workdir.child("vertices_with_neighbours");
        verticesPropertiesWithNeighbours = workdir.child("vertices_properties_with_neighbours");
        edgesWithNeighbours = workdir.child("edges_with_neighbours");

        verticesNoNeighbours = workdir.child("vertices_no_neighbours");
        verticesPropertiesNoNeighbours = workdir.child("vertices_properties_no_neighbours");
        edgesNoNeighbours = workdir.child("edges_no_neighbours");
    }

    @Override
    protected void runImpl(GraphInitTables graphInitTables) {

        dao.ytCypress().ensureDir(workdir);

        ListF<YPath> inTables = Cf.list(
                graphInitTables.mergeNeighbours,
                graphInitTables.graphTables.verticesTable,
                graphInitTables.graphTables.verticesPropertiesTable,
                graphInitTables.graphTables.edgesTable);

        LOG.info("Preparing for PutComponentsToMergeKey");
        dao.ytOps().sortAllParallel(inTables, ComponentCenter.CRYPTA_ID_KEY);

        dao.ytTr().withTransactionContext(dao, txContext -> {

            ListF<TableWithTmp> outTables = Cf.list(
                    txContext.createWithSchema(componentsToMerge, ComponentToMerge.class),
                    txContext.createWithSchema(verticesWithNeighbours, VertexInComponent.class),
                    txContext.createWithSchema(
                            verticesPropertiesWithNeighbours,
                            VertexPropertiesSchemaHelper.getUnionVertexPropertiesSchema()
                    ),
                    txContext.createWithSchema(edgesWithNeighbours, EdgeInComponent.class),
                    txContext.createWithSchema(verticesNoNeighbours, VertexInComponent.class),
                    txContext.createWithSchema(
                            verticesPropertiesNoNeighbours,
                            VertexPropertiesSchemaHelper.getUnionVertexPropertiesSchema()
                    ),
                    txContext.createWithSchema(edgesNoNeighbours, EdgeInComponent.class)
            );

            LOG.info("Running PutComponentsToMergeKey");
            dao.ytOps().reduceSync(
                    txContext.getTransactionId(),
                    inTables,
                    outTables.map(TableWithTmp::getTmpTablePath),
                    ComponentCenter.CRYPTA_ID_KEY,
                    new PutComponentsToMergeKey(params.getComponentScoringStrategy())
            );

            txContext.finalizeTables(outTables);

        });

        setYQLProtoFields(componentsToMerge);
    }

    private void setYQLProtoFields(YPath components) {
        try {
            dao.ytCypress().setAttribute(components, "_yql_proto_field_graphEngine", YQLTypeConfigGenerator.getYQlProtoFieldForGraphEngineRecord());
        } catch (java.lang.UnsatisfiedLinkError e) {
            LOG.info("Cannot find matching-model_java");
        }
    }

    @Override
    public ComponentsByMergeKeyTables getOutput() {

        return new ComponentsByMergeKeyTables(
                componentsToMerge,
                new ComponentsRawTables(
                        verticesNoNeighbours,
                        verticesPropertiesNoNeighbours,
                        edgesNoNeighbours
                ),
                new ComponentsRawTables(
                        verticesWithNeighbours,
                        verticesPropertiesWithNeighbours,
                        edgesWithNeighbours
                )
        );
    }

    @Override
    public String getDescription() {
        return "Collects merge keys for every crypta component";
    }

    public static class ComponentsByMergeKeyTables extends IterableTables {
        public YPath componentsToMerge;
        public ComponentsRawTables componentsNoNeighbours;
        public ComponentsRawTables componentsWithNeighbours;

        ComponentsByMergeKeyTables(YPath componentsToMerge,
                ComponentsRawTables componentsNoNeighbours,
                ComponentsRawTables componentsWithNeighbours) {
            this.componentsToMerge = componentsToMerge;
            this.componentsNoNeighbours = componentsNoNeighbours;
            this.componentsWithNeighbours = componentsWithNeighbours;
        }

        @Override
        public ListF<YPath> allTables() {
            return Cf.list(componentsToMerge).plus(componentsNoNeighbours.allTables()).plus(componentsWithNeighbours.allTables());
        }
    }
}
