package ru.yandex.crypta.graph2.matching.human.workflow.prepare.ops;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.crypta.graph.engine.proto.TGraph;
import ru.yandex.crypta.graph.engine.proto.TStats;
import ru.yandex.crypta.graph2.dao.yt.proto.NativeProtobufOneOfMessageEntryType;
import ru.yandex.crypta.graph2.matching.human.proto.PutComponentsToMergeKeyInRec;
import ru.yandex.crypta.graph2.matching.human.proto.PutComponentsToMergeKeyOutRec;
import ru.yandex.crypta.graph2.matching.human.workflow.merge.ops.ComponentMergeProbabilityHeuristic;
import ru.yandex.crypta.graph2.model.id.proto.IdInfo;
import ru.yandex.crypta.graph2.model.matching.component.Component;
import ru.yandex.crypta.graph2.model.matching.component.GraphInfo;
import ru.yandex.crypta.graph2.model.matching.component.score.ComponentScoringStrategy;
import ru.yandex.crypta.graph2.model.matching.proto.ComponentToMerge;
import ru.yandex.crypta.graph2.model.matching.proto.EdgeInComponent;
import ru.yandex.crypta.graph2.model.matching.proto.MergeKeyProto;
import ru.yandex.crypta.graph2.model.matching.proto.MergeNeighbour;
import ru.yandex.crypta.graph2.model.matching.proto.MergeProtoHelper;
import ru.yandex.crypta.graph2.model.matching.proto.VertexInComponent;
import ru.yandex.crypta.graph2.model.soup.edge.Edge;
import ru.yandex.crypta.graph2.model.soup.vertex.Vertex;
import ru.yandex.crypta.graph2.utils.IteratorUtils;
import ru.yandex.crypta.graph2.utils.IteratorUtils.IteratorSplit;
import ru.yandex.inside.yt.kosher.impl.operations.utils.ReducerWithKey;
import ru.yandex.inside.yt.kosher.operations.Statistics;
import ru.yandex.inside.yt.kosher.operations.Yield;
import ru.yandex.inside.yt.kosher.tables.YTableEntryType;

public class PutComponentsToMergeKey implements ReducerWithKey<PutComponentsToMergeKeyInRec,
        PutComponentsToMergeKeyOutRec, String> {

    private ComponentScoringStrategy componentScoringStrategy = null;

    public PutComponentsToMergeKey() {}

    public PutComponentsToMergeKey(ComponentScoringStrategy componentScoringStrategy) {
        this.componentScoringStrategy = componentScoringStrategy;
    }

    @Override
    public String key(PutComponentsToMergeKeyInRec rec) {
        if (rec.hasMergeNeighbourRec1()) {
            return rec.getMergeNeighbourRec1().getCryptaId();
        } else if (rec.hasVertexInComponentRec2()) {
            return rec.getVertexInComponentRec2().getCryptaId();
        } else if (rec.hasVertexPropertiesInComponentRec3()) {
            return rec.getVertexPropertiesInComponentRec3().getCryptaId();
        } else if (rec.hasEdgeInComponentsRec4()) {
            return rec.getEdgeInComponentsRec4().getCryptaId();
        } else {
            throw new IllegalArgumentException("Rec type is not supported " + rec);
        }
    }

    @Override
    public void reduce(String cryptaId,
                       IteratorF<PutComponentsToMergeKeyInRec> entries,
                       Yield<PutComponentsToMergeKeyOutRec> yield,
                       Statistics statistics) {
        IteratorSplit<PutComponentsToMergeKeyInRec> mergeKeysAndNext = IteratorUtils.takeWhile(entries,
                PutComponentsToMergeKeyInRec::hasMergeNeighbourRec1
        );

        List<MergeNeighbour> mergeNeighbours = mergeKeysAndNext.getHead()
                .stream()
                .map(PutComponentsToMergeKeyInRec::getMergeNeighbourRec1)
                .collect(Collectors.toUnmodifiableList());

        Set<MergeKeyProto> mergeKeys = mergeNeighbours
                .stream()
                .map(MergeProtoHelper::extractMergeKey)
                .collect(Collectors.toUnmodifiableSet());

        int neighboursCount = mergeKeys.size();
        int neighboursWeight = ComponentMergeProbabilityHeuristic.getComponentMergeWeight(
                mergeNeighbours.stream().map(MergeProtoHelper::edgeTypeStrength)
        );

        ComponentRecs parsedRecs = collectToComponent(mergeKeysAndNext.getTail());
        TGraph graphEngine = parsedRecs.toTGraph();
        TStats stats = TStats.newBuilder().build();

        if (componentScoringStrategy != null) {
            stats = componentScoringStrategy.stats(new Component(graphEngine), new GraphInfo());
        }

        for (MergeKeyProto mergeKey : mergeKeys) {

            ComponentToMerge graphToMerge = ComponentToMerge.newBuilder()
                    .setCryptaId(cryptaId)
                    .setMergeKey(mergeKey.getMergeKey())
                    .setMergeKeyType(mergeKey.getMergeKeyType())
                    .setNeighboursCount(neighboursCount)
                    .setNeighboursWeight(neighboursWeight)
                    .setGraphEngine(graphEngine)
                    .setStats(stats)
                    .build();

            yield.yield(PutComponentsToMergeKeyOutRec.newBuilder().setComponentToMergeRec1(graphToMerge).build());
        }

        yieldComponentAsRecs(yield, parsedRecs, neighboursCount > 0);

    }

    static class ComponentRecs {
        List<VertexInComponent> vertices = new ArrayList<>();
        List<EdgeInComponent> edges = new ArrayList<>();
        List<IdInfo> infos = new ArrayList<>();

        TGraph toTGraph() {
            Component component = new Component();
            for (var edge : edges) {
                component.addInnerEdge(new Edge(edge));
            }
            if (edges.isEmpty()) {
                if (vertices.size() == 1) {
                    component.addVertex(new Vertex(vertices.get(0)));
                } else {
                    // TODO: fix outdated multi-vertex component issue
                    // throw new IllegalStateException("At least one vertex must be defined" + vertices);
                }
            }
            GraphInfo graphInfo = new GraphInfo();
            for (var info : infos) {
                graphInfo.verticesProperties.accept(info);
            }
            return component.computeTGraph(graphInfo);
        }
    }

    private ComponentRecs collectToComponent(Iterator<PutComponentsToMergeKeyInRec> recs) {
        ComponentRecs parsedRecs = new ComponentRecs();

        while (recs.hasNext()) {
            PutComponentsToMergeKeyInRec rec = recs.next();

            if (rec.hasVertexInComponentRec2()) {
                parsedRecs.vertices.add(rec.getVertexInComponentRec2());
            } else if (rec.hasVertexPropertiesInComponentRec3()) {
                parsedRecs.infos.add(rec.getVertexPropertiesInComponentRec3());
            } else if (rec.hasEdgeInComponentsRec4()) {
                parsedRecs.edges.add(rec.getEdgeInComponentsRec4());
            } else {
                throw new IllegalArgumentException("Rec type is not supported " + rec);
            }
        }
        return parsedRecs;
    }


    private void yieldComponentAsRecs(Yield<PutComponentsToMergeKeyOutRec> yield, ComponentRecs graph,
                                      boolean hasNeighbours) {
        if (graph.edges.isEmpty()) {
            if (graph.vertices.size() == 1) {
                VertexInComponent singleVertex = graph.vertices.get(0);
                if (hasNeighbours) {
                    yield.yield(PutComponentsToMergeKeyOutRec.newBuilder().setVertexWithMergeKey2(singleVertex).build());
                } else {
                    yield.yield(PutComponentsToMergeKeyOutRec.newBuilder().setVertexNoMergeKey5(singleVertex).build());
                }
            } else {
                // TODO: fix outdated multi-vertex component issue
                // throw new IllegalStateException("Empty edges imply single vertex component " + graph.vertices);
            }
        } else {
            for (EdgeInComponent edge : graph.edges) {
                if (hasNeighbours) {
                    yield.yield(PutComponentsToMergeKeyOutRec.newBuilder().setEdgeWithMergeKey4(edge).build());
                } else {
                    yield.yield(PutComponentsToMergeKeyOutRec.newBuilder().setEdgeNoMergeKey7(edge).build());
                }
            }
        }

        for (IdInfo idInfo : graph.infos) {
            if (hasNeighbours) {
                yield.yield(PutComponentsToMergeKeyOutRec.newBuilder().setIdInfoWithMergeKey3(idInfo).build());
            } else {
                yield.yield(PutComponentsToMergeKeyOutRec.newBuilder().setIdInfoNoMergeKey6(idInfo).build());
            }
        }
    }

    @Override
    public YTableEntryType<PutComponentsToMergeKeyInRec> inputType() {
        return new NativeProtobufOneOfMessageEntryType<>(PutComponentsToMergeKeyInRec.newBuilder(), false);
    }

    @Override
    public YTableEntryType<PutComponentsToMergeKeyOutRec> outputType() {
        return new NativeProtobufOneOfMessageEntryType<>(PutComponentsToMergeKeyOutRec.newBuilder(), false);
    }
}
