package ru.yandex.crypta.graph2.model.matching.component;

import java.util.HashMap;
import java.util.stream.Stream;

import com.google.protobuf.InvalidProtocolBufferException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.crypta.graph.Identifier;
import ru.yandex.crypta.graph.engine.proto.TEdge;
import ru.yandex.crypta.graph.engine.proto.TGraph;
import ru.yandex.crypta.graph.engine.proto.TIdsInfo;
import ru.yandex.crypta.graph.engine.proto.TSocdemInfo;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.graph.soup.config.proto.ELogSourceType;
import ru.yandex.crypta.graph.soup.config.proto.ESourceType;
import ru.yandex.crypta.graph2.model.matching.score.MetricsTree;
import ru.yandex.crypta.graph2.model.soup.edge.Edge;
import ru.yandex.crypta.graph2.model.soup.props.DeviceId;
import ru.yandex.crypta.graph2.model.soup.props.UaProfile;
import ru.yandex.crypta.graph2.model.soup.props.VertexPropertiesCollector;
import ru.yandex.crypta.graph2.model.soup.props.Yandexuid;
import ru.yandex.crypta.graph2.model.soup.props.info.ExactSocdem;
import ru.yandex.crypta.graph2.model.soup.vertex.Vertex;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.crypta.lib.proto.identifiers.TGenericID;
import ru.yandex.inside.yt.kosher.impl.ytree.object.FieldsBindingStrategy;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;

@YTreeObject(bindingStrategy = FieldsBindingStrategy.ANNOTATED_ONLY)
public class Component {

    @YTreeField
    private String cryptaId;
    @YTreeField
    private int neighboursCount = 0;
    @YTreeField
    private int neighboursWeight = 0;
    @YTreeField
    private SetF<Vertex> vertices = Cf.hashSet();
    @YTreeField
    private SetF<Edge> innerEdges = Cf.hashSet();

    @YTreeField
    private MetricsTree scoreTree;


    // pre-compute these to speed up
    private TGraph graph = null;

    public Component(TGraph graph) {
        this.graph = graph;
    }

    public Component() {

    }

    public Component(Vertex vertex) {
        this.addVertex(vertex);
    }

    public Component(SetF<Vertex> vertices) {
        this.vertices = vertices;
    }

    public Component(String cryptaId) {
        this.cryptaId = cryptaId;
    }

    public Component merge(Component toMerge, ListF<Edge> byEdges) {
        Component newComponent = this.copy();
        newComponent.addAllVertices(toMerge.getVertices());
        newComponent.addInnerEdges(toMerge.getInnerEdges());
        newComponent.addInnerEdges(byEdges);
        return newComponent;
    }

    public Component mergeWithGraphEngine(Component toMerge, ListF<Edge> byEdges) {
        Component newComponent = this.merge(toMerge, byEdges);

        newComponent.setGraphEngine(mergeDistinct(graph, toMerge.getGraphEngine(), byEdges));
        return newComponent;
    }

    public void setGraphEngine(TGraph graph) {
        this.graph = graph;
    }

    public TGraph getGraphEngine() {
        return graph;
    }

    public static TGraph mergeDistinct(TGraph first, TGraph second, ListF<Edge> byEdges) {
        TGraph.Builder graph = TGraph.newBuilder(first);
        int firstSize = graph.getVerticesCount();

        graph.mergeIdsInfo(second.getIdsInfo());
        graph.addAllVertices(second.getVerticesList());

        // shift indexes for second graph
        for (TEdge secondEdge : second.getEdgesList()) {
            TEdge.Builder edge = TEdge.newBuilder(secondEdge);
            edge.setVertex1(edge.getVertex1() + firstSize);
            edge.setVertex2(edge.getVertex2() + firstSize);
            graph.addEdges(edge);
        }

        // add edge between
        for (Edge edge : byEdges) {
            int firstIndex = 0;
            int secondIndex = 0;
            TGenericID firstVertex = null;
            TGenericID secondVertex = null;
            try {
                firstVertex = convertVertexToGenericId(edge.getVertex1());
                secondVertex = convertVertexToGenericId(edge.getVertex2());
            } catch (RuntimeException | InvalidProtocolBufferException e) {
                e.printStackTrace();
                continue;
            }
            if (firstVertex == null || secondVertex == null) {
                continue;
            }

            int index = 0;
            for (TGenericID vertex : first.getVerticesList()) {
                if (vertex.equals(firstVertex)) {
                    firstIndex = index;
                }
                if (vertex.equals(secondVertex)) {
                    secondIndex = index;
                }
                index++;
            }
            index = firstSize;
            for (TGenericID vertex : second.getVerticesList()) {
                if (vertex.equals(firstVertex)) {
                    firstIndex = index;
                }
                if (vertex.equals(secondVertex)) {
                    secondIndex = index;
                }
                index++;
            }
            TEdge.Builder edgeEngine = TEdge.newBuilder();
            edgeEngine
                    .setVertex1(firstIndex)
                    .setVertex2(secondIndex)
                    .setLogSource(ELogSourceType.valueOf(edge.getLogSource().name()))
                    .setSourceType(ESourceType.valueOf(edge.getSourceType().name()))
                    .setDatesWeight(edge.getDatesWeight().getOrElse(0.))
                    .setSurvivalWeight(edge.getSurvivalWeight().getOrElse(0.))
                    .setIndevice(edge.getIndevice().getOrElse(false));
            graph.addEdges(edgeEngine);
        }

        return graph.build();
    }

    public void addVertex(Vertex v) {
        this.vertices.add(v);
        graph = null;
    }

    public void addAllVertices(CollectionF<Vertex> vertices) {
        this.vertices.addAll(vertices);
        graph = null;
    }

    public int size() {
        if (graph != null && graph.getVerticesCount() > 0) {
            return graph.getVerticesCount();
        }
        return vertices.size();
    }

    public SetF<Vertex> getVertices() {
        return vertices;
    }

    public void setVertices(SetF<Vertex> vertices) {
        this.vertices = vertices;
        graph = null;
    }

    public String getCryptaId() {
        return cryptaId;
    }

    public void setCryptaId(String cryptaId) {
        this.cryptaId = cryptaId;
    }

    public boolean hasVertex(Vertex vertex) {
        return this.vertices.containsTs(vertex);
    }

    public SetF<Edge> getInnerEdges() {
        return innerEdges;
    }

    public void setInnerEdges(SetF<Edge> innerEdges) {
        this.innerEdges = innerEdges;
        graph = null;
    }

    public void addInnerEdge(Edge edge) {
        this.innerEdges.add(edge);
        this.vertices.add(edge.getVertex1());
        this.vertices.add(edge.getVertex2());
        graph = null;
    }

    public void addInnerEdges(Stream<Edge> edges) {
        edges.forEach(this::addInnerEdge);
    }

    public void addInnerEdges(CollectionF<Edge> edges) {
        for (Edge edge : edges) {
            addInnerEdge(edge);
        }
    }

    public int getNeighboursCount() {
        return neighboursCount;
    }

    public void setNeighboursCount(int neighboursCount) {
        this.neighboursCount = neighboursCount;
    }

    public int getNeighboursWeight() {
        return neighboursWeight;
    }

    public void setNeighboursWeight(int neighboursWeight) {
        this.neighboursWeight = neighboursWeight;
    }

    public MetricsTree getScoreTree() {
        return scoreTree;
    }

    public void setScoreTree(MetricsTree scoreTree) {
        this.scoreTree = scoreTree;
    }

    public TGraph computeTGraph() {
        return computeTGraph(new GraphInfo());
    }

    static public TGenericID convertVertexToGenericId(Vertex vertex) throws InvalidProtocolBufferException {
        Identifier id = new Identifier(vertex.getIdType(), vertex.getId());
        return id.toTGenericID();
    }

    public TGraph computeTGraph(GraphInfo graphInfo) {
        if (graph != null && graph.getVerticesCount() > 0) {
            // important part
            return graph;
        }

        TGraph.Builder graph = TGraph.newBuilder();
        if (cryptaId != null) {
            graph.setId(Long.parseUnsignedLong(cryptaId));
        }
        TIdsInfo.Builder idsInfo = TIdsInfo.newBuilder();
        MapF<Vertex, DeviceId> deviceIdsInfo = graphInfo.verticesProperties.getDeviceIds();
        MapF<Vertex, Yandexuid> yuidsInfo = graphInfo.verticesProperties.getYuids();

        HashMap<Vertex, Integer> vertexToIndex = new HashMap<>();
        int index = 0;

        TGenericID genericID = null;
        for (Vertex vertex : this.vertices) {
            try {
                genericID = convertVertexToGenericId(vertex);
                vertexToIndex.put(vertex, index);
                index++;

            } catch (RuntimeException | InvalidProtocolBufferException e) {
                continue;
            }
            graph.addVertices(genericID);


            if (Soup.CONFIG.isDeviceIdMainId(vertex.getIdType()) && deviceIdsInfo.containsKeyTs(vertex)) {
                DeviceId deviceId = deviceIdsInfo.getTs(vertex);
                int mainRegion = deviceId.getMainRegion().getOrElse(0);
                idsInfo.addDevicesInfoBuilder()
                        .setId(genericID)
                        .setIsActive(deviceId.isActive())
                        .setMainRegion(mainRegion);
            }
            if (vertex.getIdType() == EIdType.YANDEXUID && yuidsInfo.containsKeyTs(vertex)) {
                Yandexuid yuid = yuidsInfo.getTs(vertex);
                int mainRegion = yuid.getMainRegion().getOrElse(0);
                boolean isMobile = !new UaProfile(yuid.getUaProfile()).isDesktop();
                boolean isActive = yuid.isActive();
                if (mainRegion > 0 || (isActive && isMobile)) {
                    idsInfo.addBrowsersInfoBuilder()
                            .setId(genericID)
                            .setIsMobile(isMobile)
                            .setIsActive(isActive)
                            .setMainRegion(mainRegion);
                }
            }
        }

        for (Edge edge : this.innerEdges) {
            if (!vertexToIndex.containsKey(edge.getVertex1()) || !vertexToIndex.containsKey(edge.getVertex2())) {
                continue;
            }
            TEdge protoEdge = TEdge.newBuilder()
                    .setVertex1(vertexToIndex.get(edge.getVertex1()))
                    .setVertex2(vertexToIndex.get(edge.getVertex2()))
                    .setLogSource(ELogSourceType.valueOf(edge.getLogSource().name()))
                    .setSourceType(ESourceType.valueOf(edge.getSourceType().name()))
                    .setDatesWeight(edge.getDatesWeight().getOrElse(0.))
                    .setSurvivalWeight(edge.getSurvivalWeight().getOrElse(0.))
                    .setIndevice(edge.getIndevice().getOrElse(false))
                    .build();
            graph.addEdges(protoEdge);
        }

        MapF<Vertex, ListF<ExactSocdem>> socdems = graphInfo.verticesProperties.getExactSocdems();
        for (Vertex vertex : socdems.keySet().intersect(vertices)) {
            TGenericID id;
            try {
                id = convertVertexToGenericId(vertex);
            } catch (InvalidProtocolBufferException e) {
                continue;
            }
            for (var socdem : socdems.get(vertex)) {
                TSocdemInfo.Builder socdemInfoBuilder = idsInfo.addSocdemInfoBuilder();
                socdemInfoBuilder.setId(id);
                if (socdem.getGender().isPresent()) {
                    socdemInfoBuilder.setGender(socdem.getGender().get().value());
                }
                if (socdem.getSocdemSource().isPresent()) {
                    socdemInfoBuilder.setSocdemSource(socdem.getSocdemSource().get());
                }
                if (socdem.getYearOfBirth().isPresent()) {
                    socdemInfoBuilder.setYearOfBirth(socdem.getYearOfBirth().get());
                }
            }
        }
        graph.setIdsInfo(idsInfo);

        this.graph = graph.build();
        return this.graph;
    }

    public ListF<ExactSocdem> getExactSocdems() {
        if (graph != null && graph.getVerticesCount() > 0) {
            ListF<ExactSocdem> socdems = Cf.arrayList();
            for (var socdem : graph.getIdsInfo().getSocdemInfoList()) {
                socdems.add(VertexPropertiesCollector.getExactSocdem(
                        socdem.getGender(),
                        (int)socdem.getYearOfBirth(),
                        socdem.getSocdemSource())
                );
            }
            return socdems;
        }
        return Cf.list();
    }

    public ListF<ExactSocdem> getExactSocdemsFromVP(VertexPropertiesCollector verticesProperties) {
        return vertices
                .filterMap(v -> verticesProperties.getExactSocdems().getO(v))
                .flatten();
    }

    public ListF<ExactSocdem> getExactSocdems(VertexPropertiesCollector verticesProperties) {
        ListF<ExactSocdem> result = getExactSocdems();
        if (result.size() > 0) {
            return result;
        }
        return getExactSocdemsFromVP(verticesProperties);
    }

    public ListF<Integer> getRegions() {
        if (graph != null && graph.getVerticesCount() > 0) {
            ListF<Integer> regions = Cf.arrayList();
            for (var info : graph.getIdsInfo().getBrowsersInfoList()) {
                if (info.getMainRegion() > 0) {
                    regions.add((int)info.getMainRegion());
                }
            }
            for (var info : graph.getIdsInfo().getDevicesInfoList()) {
                if (info.getMainRegion() > 0) {
                    regions.add((int)info.getMainRegion());
                }
            }
            return regions;
        }
        return Cf.list();
    }

    public ListF<Integer> getRegionsFromVP(VertexPropertiesCollector verticesProperties) {
        ListF<Integer> yuidRegions = vertices
                .filterMap(v -> verticesProperties.getYuids().getO(v))
                .filterMap(Yandexuid::getMainRegion);

        ListF<Integer> deviceIdsRegions = vertices
                .filterMap(v -> verticesProperties.getDeviceIds().getO(v))
                .filterMap(DeviceId::getMainRegion);
        return yuidRegions.plus(deviceIdsRegions);
    }

    public ListF<Integer> getRegions(VertexPropertiesCollector verticesProperties) {
        ListF<Integer> result = getRegions();
        if (result.size() > 0) {
            return result;
        }
        return getRegionsFromVP(verticesProperties);
    }

    public Component copy() {
        Component copy = new Component();
        copy.cryptaId = this.cryptaId;
        copy.vertices = Cf.toHashSet(this.vertices);
        copy.innerEdges = Cf.toHashSet(this.innerEdges);
        return copy;
    }

    @Override
    public String toString() {
        return "Component{" +
                "cryptaId=" + cryptaId +
                ", vertices=" + vertices +
                '}';
    }
}
