package ru.yandex.crypta.graph2.model.matching.component;

import java.util.Comparator;
import java.util.Objects;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph.soup.config.Soup;
import ru.yandex.crypta.graph2.dao.yt.schema.extractor.CustomColumnType;
import ru.yandex.crypta.graph2.model.soup.vertex.ProtobufEnumIdTypeSerializer;
import ru.yandex.crypta.graph2.model.soup.vertex.Vertex;
import ru.yandex.crypta.graph2.utils.HashUtils;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;
import ru.yandex.inside.yt.kosher.impl.ytree.object.FieldsBindingStrategy;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeSerializerClass;
import ru.yandex.yt.ytclient.tables.ColumnValueType;

@YTreeObject(bindingStrategy = FieldsBindingStrategy.ANNOTATED_ONLY)
public class ComponentCenter implements Comparable<ComponentCenter> {

    public static final String CC_ID_COLUMN = "ccId";
    public static final String CC_ID_TYPE_COLUMN = "ccIdType";
    public static final String CC_NEIGHBOURS_COUNT_COLUMN = "ccNeighboursCount";
    public static final String CRYPTA_ID_COLUMN = "cryptaId";

    public static final ListF<String> CRYPTA_ID_KEY = Cf.list(CRYPTA_ID_COLUMN);

    public static final Comparator<ComponentCenter> COMPARATOR = Comparator
            .comparing(ComponentCenter::getCcId)
            .thenComparing(v -> Soup.CONFIG.name(v.getCcIdType()));

    @YTreeField
    private String ccId;

    @YTreeField
    @YTreeSerializerClass(ProtobufEnumIdTypeSerializer.class)
    @CustomColumnType(ColumnValueType.STRING)
    private EIdType ccIdType;
    @YTreeField
    private String cryptaId;
    @YTreeField
    private Option<Integer> ccNeighboursCount = Option.empty();

    public ComponentCenter(String ccId, EIdType idType) {
        this(ccId, idType, Option.empty());
    }

    public ComponentCenter(String ccId, EIdType ccIdType, int ccNeighboursCount) {
        this(ccId, ccIdType, Option.of(ccNeighboursCount));
    }

    public ComponentCenter(String ccId, EIdType ccIdType, Option<Integer> ccNeighboursCount) {
        this.ccId = ccId;
        this.ccIdType = ccIdType;
        this.ccNeighboursCount = ccNeighboursCount;

        this.cryptaId = toCryptaId(ccId, ccIdType);
    }

    public static String toCryptaId(String ccId, EIdType ccIdType) {
        String cryptaIdStr = ccId + "(" + Soup.CONFIG.name(ccIdType) + ")";
        return HashUtils.arcadiaCompatibleMurmurUint64String(cryptaIdStr);
    }

    public static ComponentCenter fromVertex(Vertex v) {
        return new ComponentCenter(v.getId(), v.getIdType());
    }

    public static Vertex toVertex(ComponentCenter cc) {
        return new Vertex(cc.getCcId(), cc.getCcIdType());
    }


    public String getCcId() {
        return ccId;
    }

    public EIdType getCcIdType() {
        return ccIdType;
    }

    public Option<Integer> getCcNeighboursCount() {
        return ccNeighboursCount;
    }

    public void setCcNeighboursCount(int ccNeighboursCount) {
        this.ccNeighboursCount = Option.of(ccNeighboursCount);
    }

    public Vertex convertToVertex() {
        return ComponentCenter.toVertex(this);
    }

    public static ComponentCenter danger(String cryptaId) {
        return new ComponentCenter(cryptaId, EIdType.CRYPTA_ID);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof ComponentCenter that)) {
            return false;
        }
        return Objects.equals(ccId, that.ccId) &&
                ccIdType == that.ccIdType;
    }

    @Override
    public int hashCode() {
        return Objects.hash(ccId, ccIdType);
    }

    @Override
    public String toString() {
        return ccId + "(" + ccIdType + ")";
    }

    public String getCryptaId() {
        return cryptaId;
    }

    @Override
    public int compareTo(ComponentCenter o) {
        return COMPARATOR.compare(this, o);
    }
}

