package ru.yandex.crypta.graph2.model.matching.component.score;

import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.model.matching.component.Component;
import ru.yandex.crypta.graph2.model.matching.component.GraphInfo;
import ru.yandex.crypta.graph2.model.matching.component.score.extractors.CrossDeviceCountExtractor;
import ru.yandex.crypta.graph2.model.matching.component.score.extractors.DevicesCountExtractor;
import ru.yandex.crypta.graph2.model.matching.component.score.extractors.EmailsCountExtractor;
import ru.yandex.crypta.graph2.model.matching.component.score.extractors.EmailsLcsExtractor;
import ru.yandex.crypta.graph2.model.matching.component.score.extractors.IdTypeVerticesCountExtractor;
import ru.yandex.crypta.graph2.model.matching.component.score.extractors.LoginsCountExtractor;
import ru.yandex.crypta.graph2.model.matching.component.score.extractors.LoginsLcsExtractor;
import ru.yandex.crypta.graph2.model.matching.score.MetricsTree;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;

public class HumanMultiHistogramScoringStrategy implements ComponentScoringStrategy {

    protected WeightedMultiScoringStrategy weighted;

    public HumanMultiHistogramScoringStrategy() {
        this(false, Option.empty());
    }

    public HumanMultiHistogramScoringStrategy(boolean requireYandexuidForPersonalIds, Option<Double> crossDeviceWeight) {

        HistogramCountsScoringStrategy devicesCountHist = HistogramCountsScoringStrategy
                .metric(
                        "devices_count", "Count of devices"
                ).scoringCount(
                        new DevicesCountExtractor(true)
                ).lessOrEqualAs(0, 0)
                .lessOrEqualAs(1, 0.6)
                .lessOrEqualAs(3, 0.3)
                .uniformDecreasingRange(4, 20, 0.1)
                .andTheRestAs(0)
                .build();

        HistogramCountsScoringStrategy loginsCountHist = HistogramCountsScoringStrategy
                .metric(
                        "logins_count", "Count of logins"
                ).scoringCount(
                        new LoginsCountExtractor(requireYandexuidForPersonalIds)
                ).lessOrEqualAs(0, 0)
                .lessOrEqualAs(2, 0.6)
                .lessOrEqualAs(3, 0.3)
                .uniformDecreasingRange(4, 20, 0.1)
                .andTheRestAs(0)
                .build();

        HistogramCountsScoringStrategy emailsCountHist = HistogramCountsScoringStrategy
                .metric(
                        "emails_count", "Count of emails"
                ).scoringCount(
                        new EmailsCountExtractor(requireYandexuidForPersonalIds)
                ).lessOrEqualAs(0, 0)
                .lessOrEqualAs(3, 0.6)
                .lessOrEqualAs(5, 0.3)
                .uniformDecreasingRange(6, 20, 0.1)
                .andTheRestAs(0)
                .build();

        HistogramCountsScoringStrategy phonesCountHist = HistogramCountsScoringStrategy
                .metric(
                        "phones_count", "Count of phones"
                ).scoringCount(
                        new IdTypeVerticesCountExtractor(EIdType.PHONE)
                ).lessOrEqualAs(0, 0)
                .lessOrEqualAs(2, 0.6)
                .lessOrEqualAs(3, 0.3)
                .uniformDecreasingRange(4, 20, 0.1)
                .andTheRestAs(0)
                .build();

        HistogramCountsScoringStrategy vkCountHist = HistogramCountsScoringStrategy
                .metric(
                        "vk_count", "Count of vk ids"
                ).scoringCount(
                        new IdTypeVerticesCountExtractor(EIdType.VK_ID)
                ).lessOrEqualAs(0, 0)
                .lessOrEqualAs(1, 0.6)
                .lessOrEqualAs(2, 0.3)
                .uniformDecreasingRange(3, 20, 0.1)
                .andTheRestAs(0)
                .build();

        HistogramCountsScoringStrategy okCountHist = HistogramCountsScoringStrategy
                .metric(
                        "ok_count", "Count of ok ids"
                ).scoringCount(
                        new IdTypeVerticesCountExtractor(EIdType.OK_ID)
                ).lessOrEqualAs(0, 0)
                .lessOrEqualAs(1, 0.6)
                .lessOrEqualAs(2, 0.3)
                .uniformDecreasingRange(3, 20, 0.1)
                .andTheRestAs(0)
                .build();

        HistogramCountsScoringStrategy verticesCountHist = HistogramCountsScoringStrategy
                .metric(
                        "vertices_count", "Total count of vertices in component"
                ).scoringCount(
                        (cp, graphInfo) -> cp.getVertices().size()
                ).lessOrEqualAs(0, 0)
                .uniformIncreasingRange(0, 1000, 0.7)
                .uniformDecreasingRange(1001, 10000, 0.3)
                .andTheRestAs(0)
                .build();

        HistogramCountsScoringStrategy crossDeviceHist = HistogramCountsScoringStrategy
                .metric(
                        "cross_device", "Cross device component"
                ).scoringCount(
                        (cp, graphInfo) -> new CrossDeviceCountExtractor().apply(cp, graphInfo)
                ).lessOrEqualAs(0, 0)
                .andTheRestAs(1)
                .build();

        HistogramCountsScoringStrategy loginsLcsHist = HistogramCountsScoringStrategy
                .metric(
                        "logins_lcs", "Longest common substring of logins"
                ).scoringCount(
                        new LoginsLcsExtractor()
                ).lessOrEqualAs(1, 0)
                .lessOrEqualAs(6, 0.12)
                .lessOrEqualAs(12, 0.19)
                .lessOrEqualAs(18, 0.29)
                .andTheRestAs(0.4)
                .build();  // https://a.yandex-team.ru/arc/trunk/arcadia/junk/daflight/CRYPTR_1256/antispam_logins.ipynb

        HistogramCountsScoringStrategy emailsLcsHist = HistogramCountsScoringStrategy
                .metric(
                        "emails_lcs", "Longest common substring of emails"
                ).scoringCount(
                        new EmailsLcsExtractor()
                ).lessOrEqualAs(1, 0)
                .lessOrEqualAs(7, 0.11)
                .lessOrEqualAs(14, 0.19)
                .lessOrEqualAs(21, 0.29)
                .andTheRestAs(0.41)
                .build();  // https://a.yandex-team.ru/arc/trunk/arcadia/junk/daflight/CRYPTR_1256/antispam_emails.ipynb

        this.weighted = new WeightedMultiScoringStrategy()
                .weighting(devicesCountHist, 1)
                .weighting(loginsCountHist, 1)
                .weighting(emailsCountHist, 1)
                .weighting(loginsLcsHist, 1)
                .weighting(emailsLcsHist, 1)
                .weighting(phonesCountHist, 1)
                .weighting(vkCountHist, 1)
                .weighting(okCountHist, 1)
                .weighting(verticesCountHist, 100)
                .weighting(crossDeviceHist, crossDeviceWeight);
    }

    @Override
    public MetricsTree scoreTree(Component component, GraphInfo graphInfo) {
        return weighted.scoreTree(component, graphInfo);
    }

    @Override
    public String getName() {
        return "human";
    }
}
