package ru.yandex.crypta.graph2.model.matching.component.score;

import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.model.matching.component.Component;
import ru.yandex.crypta.graph2.model.matching.component.GraphInfo;
import ru.yandex.crypta.graph2.model.matching.component.score.extractors.CrossDeviceCountExtractor;
import ru.yandex.crypta.graph2.model.matching.component.score.extractors.DevicesCountExtractor;
import ru.yandex.crypta.graph2.model.matching.component.score.extractors.EmailsCountExtractor;
import ru.yandex.crypta.graph2.model.matching.component.score.extractors.IdTypeVerticesCountExtractor;
import ru.yandex.crypta.graph2.model.matching.component.score.extractors.LoginsCountExtractor;
import ru.yandex.crypta.graph2.model.matching.score.MetricsTree;
import ru.yandex.crypta.lib.proto.identifiers.EIdType;

public class TrueHumanMultiHistogramScoringStrategy implements ComponentScoringStrategy {

    protected WeightedMultiScoringStrategy weighted;


    public TrueHumanMultiHistogramScoringStrategy() {
        this(false, Option.empty());
    }

    public TrueHumanMultiHistogramScoringStrategy(boolean requireYandexuidForPersonalIds, Option<Double> crossDeviceWeight) {

        TrueHistogramCountsScoringStrategy devicesCountHist = TrueHistogramCountsScoringStrategy
                .metric(
                        "probability_devices_count", "Count of devices probability histogram"
                ).scoringCount(
                        new DevicesCountExtractor(true)
                ).lessOrEqualAs(0, 0)
                .lessOrEqualAs(1, 0.6)
                .lessOrEqualAs(3, 0.3)
                .uniformDecreasingRange(4, 20, 0.1)
                .andTheRest()
                .build();

        TrueHistogramCountsScoringStrategy loginsCountHist = TrueHistogramCountsScoringStrategy
                .metric(
                        "probability_logins_count", "Count of logins probability histogram"
                ).scoringCount(
                        new LoginsCountExtractor(requireYandexuidForPersonalIds)
                ).lessOrEqualAs(0, 0)
                .lessOrEqualAs(2, 0.6)
                .lessOrEqualAs(3, 0.3)
                .uniformDecreasingRange(4, 20, 0.1)
                .andTheRest()
                .build();

        TrueHistogramCountsScoringStrategy emailsCountHist = TrueHistogramCountsScoringStrategy
                .metric(
                        "probability_emails_count", "Count of emails probability histogram"
                ).scoringCount(
                        new EmailsCountExtractor(requireYandexuidForPersonalIds)
                ).lessOrEqualAs(0, 0)
                .lessOrEqualAs(3, 0.6)
                .lessOrEqualAs(5, 0.3)
                .uniformDecreasingRange(6, 20, 0.1)
                .andTheRest()
                .build();

        TrueHistogramCountsScoringStrategy phonesCountHist = TrueHistogramCountsScoringStrategy
                .metric(
                        "probability_phones_count", "Count of phones probability histogram"
                ).scoringCount(
                        new IdTypeVerticesCountExtractor(EIdType.PHONE)
                ).lessOrEqualAs(0, 0)
                .lessOrEqualAs(2, 0.6)
                .lessOrEqualAs(3, 0.3)
                .uniformDecreasingRange(4, 20, 0.1)
                .andTheRest()
                .build();

        TrueHistogramCountsScoringStrategy vkCountHist = TrueHistogramCountsScoringStrategy
                .metric(
                        "probability_vk_count", "Count of vk ids probability histogram"
                ).scoringCount(
                        new IdTypeVerticesCountExtractor(EIdType.VK_ID)
                ).lessOrEqualAs(0, 0)
                .lessOrEqualAs(1, 0.6)
                .lessOrEqualAs(2, 0.3)
                .uniformDecreasingRange(3, 20, 0.1)
                .andTheRest()
                .build();

        TrueHistogramCountsScoringStrategy okCountHist = TrueHistogramCountsScoringStrategy
                .metric(
                        "probability_ok_count", "Count of ok ids probability histogram"
                ).scoringCount(
                        new IdTypeVerticesCountExtractor(EIdType.OK_ID)
                ).lessOrEqualAs(0, 0)
                .lessOrEqualAs(1, 0.6)
                .lessOrEqualAs(2, 0.3)
                .uniformDecreasingRange(3, 20, 0.1)
                .andTheRest()
                .build();

        TrueHistogramCountsScoringStrategy verticesCountHist = TrueHistogramCountsScoringStrategy
                .metric(
                        "probability_vertices_count",
                        "Total count of vertices in component probability histogram"
                ).scoringCount(
                        (cp, graphInfo) -> cp.getVertices().size()
                ).lessOrEqualAs(0, 0)
                .uniformIncreasingRange(0, 1000, 0.7)
                .uniformDecreasingRange(1001, 10000, 0.3)
                .andTheRest()
                .build();

        TrueHistogramCountsScoringStrategy crossDeviceHist = TrueHistogramCountsScoringStrategy
                .metric(
                        "probability_cross_device", "Cross device component probability histogram"
                ).scoringCount(
                        (cp, graphInfo) -> new CrossDeviceCountExtractor().apply(cp, graphInfo)
                ).lessOrEqualAs(0, 0)
                .andTheRest()
                .build();

        this.weighted = new WeightedMultiScoringStrategy()
                .weighting(devicesCountHist, 1)
                .weighting(loginsCountHist, 1)
                .weighting(emailsCountHist, 1)
                .weighting(phonesCountHist, 1)
                .weighting(vkCountHist, 1)
                .weighting(okCountHist, 1)
                .weighting(verticesCountHist, 100)
                .weighting(crossDeviceHist, crossDeviceWeight);
    }

    @Override
    public MetricsTree scoreTree(Component component, GraphInfo graphInfo) {
        return weighted.scoreTree(component, graphInfo);
    }

    @Override
    public String getName() {
        return "human";
    }
}
