package ru.yandex.crypta.graph2.model.matching.merge;

import java.util.Comparator;
import java.util.Objects;

import com.google.common.base.Splitter;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.model.soup.edge.MultiEdgeKey;
import ru.yandex.crypta.graph2.model.soup.vertex.Vertex;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.misc.lang.Check;


@YTreeObject
public class MergeKey implements Comparable<MergeKey> {

    public static final MergeKey EMPTY = new MergeKey(Option.empty(), Option.empty());
    public static final String MERGE_KEY_COLUMN = "merge_key";
    public static final String MERGE_KEY_TYPE_COLUMN = "merge_key_type";
    public static final ListF<String> REDUCE_KEY = Cf.list(MERGE_KEY_COLUMN);
    private static final Comparator<MergeKey> COMPARATOR = Comparator
            .<MergeKey, String>comparing(
                    mk -> mk.mergeKey.getOrNull(),  // for empty MergeKey
                    Comparator.nullsFirst(Comparator.naturalOrder())
            )
            .thenComparing(
                    mk -> mk.mergeKeyType.getOrNull(),  // for empty MergeKey
                    Comparator.nullsFirst(Comparator.naturalOrder())
            );

    public static class CryptaIds {
        public String leftCryptaId;
        public String rightCryptaId;

        public CryptaIds(String leftCryptaId, String rightCryptaId) {
            this.leftCryptaId = leftCryptaId;
            this.rightCryptaId = rightCryptaId;
        }

        public static CryptaIds fromMergeKey(String mergeKey) {
            var recs = Splitter.on("_").splitToList(mergeKey);
            Check.isTrue(recs.size() == 2, String.format("MergeKey %s is not valid", mergeKey));
            return new CryptaIds(recs.get(0), recs.get(1));
        }

        public static CryptaIds fromMergeKey(MergeKey mergeKey) {
            return fromMergeKey(mergeKey.getMergeKey());
        }

        public MergeKey toMergeKey() {
            return toMergeKey(leftCryptaId, rightCryptaId);
        }

        public static MergeKey toMergeKey(String leftCryptaId, String rightCryptaId) {
            if (leftCryptaId.compareTo(rightCryptaId) < 0) {
                return new MergeKey(leftCryptaId + "_" + rightCryptaId, MergeKeyType.BY_EDGES_CUT);
            } else {
                return new MergeKey(rightCryptaId + "_" + leftCryptaId, MergeKeyType.BY_EDGES_CUT);
            }
        }
    }

    @YTreeField(key = MERGE_KEY_COLUMN)
    private Option<String> mergeKey;

    @YTreeField(key = MERGE_KEY_TYPE_COLUMN)
    private Option<MergeKeyType> mergeKeyType;

    public MergeKey(String mergeKey, MergeKeyType mergeKeyType) {
        this.mergeKey = Option.of(mergeKey);
        this.mergeKeyType = Option.of(mergeKeyType);
    }

    public MergeKey(String mergeKey, String mergeKeyTypeStr) {
        this.mergeKey = Option.of(mergeKey);
        if (mergeKeyTypeStr.isEmpty()) {
            this.mergeKeyType = Option.empty();
        }
        this.mergeKeyType = Option.of(MergeKeyType.valueOf(mergeKeyTypeStr));
    }

    private MergeKey(Option<String> mergeKey, Option<MergeKeyType> mergeKeyType) {
        this.mergeKey = mergeKey;
        this.mergeKeyType = mergeKeyType;
    }

    public static MergeKey fromVertex(Vertex vertex) {
        return new MergeKey(vertex.repr(), MergeKeyType.BY_VERTEX);
    }

    public static MergeKey fromEdge(MultiEdgeKey multiEdgeKey) {
        return new MergeKey(multiEdgeKey.repr(), MergeKeyType.BY_EDGE);
    }

    public static MergeKey fromEdge(ru.yandex.crypta.graph2.model.soup.proto.MultiEdgeKey multiEdgeKey) {
        String repr = multiEdgeKey.getId1() + "(" + multiEdgeKey.getId1Type() + ")-"
                + multiEdgeKey.getId2() + "(" + multiEdgeKey.getId2Type() + ")";
        return new MergeKey(repr, MergeKeyType.BY_EDGE);
    }

    public static MergeKey betweenComponents(String c1, String c2) {
        return CryptaIds.toMergeKey(c1, c2);
    }

    public CryptaIds extractCryptaIds() {
        return CryptaIds.fromMergeKey(this);
    }

    public static MergeKey fromComponent(String cryptaId) {
        return new MergeKey(cryptaId, MergeKeyType.BY_COMPONENT);
    }

    public boolean isPresent() {
        return mergeKey.isPresent() && mergeKeyType.isPresent();
    }

    public String getMergeKey() {
        return mergeKey.get();
    }

    public MergeKeyType getMergeKeyType() {
        return mergeKeyType.get();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof MergeKey mergeKey1)) {
            return false;
        }
        return Objects.equals(mergeKey, mergeKey1.mergeKey) &&
                Objects.equals(mergeKeyType, mergeKey1.mergeKeyType);
    }

    @Override
    public int hashCode() {
        return Objects.hash(mergeKey, mergeKeyType);
    }

    @Override
    public int compareTo(MergeKey o) {
        return COMPARATOR.compare(this, o);
    }

    @Override
    public String toString() {
        if (mergeKey.isPresent()) {
            return mergeKey.get() + "(" + mergeKeyType.get() + ")";
        } else {
            return "EMPTY_MERGE_KEY";
        }
    }
}
