package ru.yandex.crypta.graph2.model.matching.merge;

import java.util.Comparator;
import java.util.Objects;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.crypta.graph2.model.matching.component.similarity.SimilarityResult;
import ru.yandex.crypta.graph2.model.matching.score.MetricsTree;
import ru.yandex.crypta.graph2.model.soup.edge.weight.EdgeScore;
import ru.yandex.inside.yt.kosher.impl.ytree.object.FieldsBindingStrategy;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeFlattenField;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;

/**
 * Offer to merge two cryptaIds together
 */
@YTreeObject(bindingStrategy = FieldsBindingStrategy.ANNOTATED_ONLY)
public class MergeOffer implements Comparable<MergeOffer> {

    public static final Comparator<MergeOffer> BY_MERGE_KEY_COMPARATOR = Comparator
            .comparing(MergeOffer::getMergeKey);

    @YTreeField
    @YTreeFlattenField
    private MergeKey mergeKey;

    // from to crypta ids
    @YTreeField(key = "cryptaId")
    private String fromCryptaId;
    @YTreeField
    private int fromCryptaIdComponentWeight;
    @YTreeField
    private int fromCryptaIdNeighboursCount;
    @YTreeField
    private int fromCryptaIdComponentSize;

    @YTreeField
    private String toCryptaId;
    @YTreeField
    private int toCryptaIdComponentWeight;
    @YTreeField
    private int toCryptaIdNeighboursCount;
    @YTreeField
    private int toCryptaIdComponentSize;

    // params required for later merge offer comparison
    @YTreeFlattenField
    @YTreeField
    private EdgeScore edgeScore;

    @YTreeField
    private Option<MetricsTree> scoreGain = Option.empty();

    @YTreeField
    private ListF<SimilarityResult> similarityResults = Cf.list();

    @YTreeField
    private MergeOfferStatus status;

    @YTreeField
    private boolean opposite = false;


    public MergeOffer(MergeKey mergeKey, String fromCryptaId, int fromCryptaIdComponentWeight,
                      int fromCryptaIdNeighboursCount, int fromCryptaIdComponentSize, String toCryptaId,
                      int toCryptaIdComponentWeight, int toCryptaIdNeighboursCount, int toCryptaIdComponentSize,
                      EdgeScore edgeScore, Option<MetricsTree> scoreGain, ListF<SimilarityResult> similarityResults,
                      MergeOfferStatus status, boolean opposite) {
        this.mergeKey = mergeKey;
        this.fromCryptaId = fromCryptaId;
        this.fromCryptaIdComponentWeight = fromCryptaIdComponentWeight;
        this.fromCryptaIdNeighboursCount = fromCryptaIdNeighboursCount;
        this.fromCryptaIdComponentSize = fromCryptaIdComponentSize;
        this.toCryptaId = toCryptaId;
        this.toCryptaIdComponentWeight = toCryptaIdComponentWeight;
        this.toCryptaIdNeighboursCount = toCryptaIdNeighboursCount;
        this.toCryptaIdComponentSize = toCryptaIdComponentSize;
        this.edgeScore = edgeScore;
        this.scoreGain = scoreGain;
        this.similarityResults = similarityResults;
        this.status = status;
        this.opposite = opposite;
    }

    public MergeOffer(MergeKey mergeKey, String fromCc, String toCc,
                      Option<MetricsTree> scoreGain, EdgeScore linkScore) {
        this(mergeKey, fromCc, toCc);

        this.scoreGain = scoreGain;
        this.edgeScore = linkScore;
    }

    public MergeOffer(MergeKey mergeKey, String fromCc, String toCc) {

        // Check.notEquals(fromCc, toCc);  // failed split offer can be within single component

        this.mergeKey = mergeKey;

        this.setFromCryptaId(fromCc);
        this.setToCryptaId(toCc);

    }

    public MergeKey getMergeKey() {
        return mergeKey;
    }

    public String getToCryptaId() {
        return toCryptaId;
    }

    public void setToCryptaId(String toCryptaId) {
        this.toCryptaId = toCryptaId;
    }

    public int getToCryptaIdComponentWeight() {
        return toCryptaIdComponentWeight;
    }

    public void setToCryptaIdComponentWeight(int toCryptaIdComponentWeight) {
        this.toCryptaIdComponentWeight = toCryptaIdComponentWeight;
    }

    public int getFromCryptaIdComponentWeight() {
        return fromCryptaIdComponentWeight;
    }

    public void setFromCryptaIdComponentWeight(int fromCryptaIdComponentWeight) {
        this.fromCryptaIdComponentWeight = fromCryptaIdComponentWeight;
    }

    public int getToCryptaIdNeighboursCount() {
        return toCryptaIdNeighboursCount;
    }

    public void setToCryptaIdNeighboursCount(int toCryptaIdNeighboursCount) {
        this.toCryptaIdNeighboursCount = toCryptaIdNeighboursCount;
    }

    public int getFromCryptaIdNeighboursCount() {
        return fromCryptaIdNeighboursCount;
    }

    public void setFromCryptaIdNeighboursCount(int fromCryptaIdNeighboursCount) {
        this.fromCryptaIdNeighboursCount = fromCryptaIdNeighboursCount;
    }

    public int getFromCryptaIdComponentSize() {
        return fromCryptaIdComponentSize;
    }

    public void setFromCryptaIdComponentSize(int fromCryptaIdComponentSize) {
        this.fromCryptaIdComponentSize = fromCryptaIdComponentSize;
    }

    public int getToCryptaIdComponentSize() {
        return toCryptaIdComponentSize;
    }

    public void setToCryptaIdComponentSize(int toCryptaIdComponentSize) {
        this.toCryptaIdComponentSize = toCryptaIdComponentSize;
    }

    public String getFromCryptaId() {
        return fromCryptaId;
    }

    public void setFromCryptaId(String fromCryptaId) {
        this.fromCryptaId = fromCryptaId;
    }

    public double getScoreGainValue() {
        return scoreGain.map(MetricsTree::getScore).getOrElse(0.0);
    }

    public Option<MetricsTree> getScoreGain() {
        return scoreGain;
    }

    public void setScoreGain(MetricsTree scoreGain) {
        this.scoreGain = Option.of(scoreGain);
    }

    public void setEdgeScore(EdgeScore edgeScore) {
        this.edgeScore = edgeScore;
    }

    public EdgeScore getEdgeScore() {
        return edgeScore;
    }

    public MergeOfferStatus getStatus() {
        return status;
    }

    public MergeOffer withStatus(MergeOfferStatus status) {
        this.status = status;
        return this;
    }

    public void setSimilarityResults(ListF<SimilarityResult> similarityResults) {
        this.similarityResults = similarityResults;
    }

    public boolean isOpposite() {
        return opposite;
    }

    public MergeOffer opposite() {
        return new MergeOffer(
                mergeKey,
                toCryptaId,
                toCryptaIdComponentWeight,
                toCryptaIdNeighboursCount,
                toCryptaIdComponentSize,
                fromCryptaId,
                fromCryptaIdComponentWeight,
                fromCryptaIdNeighboursCount,
                fromCryptaIdComponentSize,
                edgeScore,
                scoreGain,
                similarityResults,
                status,
                !opposite
        );
    }

    public MergeOffer copyWithChangedCryptaId(String fromCryptaId) {
        return copyWithChangedCryptaIds(fromCryptaId, getToCryptaId());
    }

    public MergeOffer copyWithChangedCryptaIds(String fromCryptaId, String toCryptaId) {
        return new MergeOffer(
                mergeKey,
                fromCryptaId,
                fromCryptaIdComponentWeight,
                fromCryptaIdNeighboursCount,
                fromCryptaIdComponentSize,
                toCryptaId,
                toCryptaIdComponentWeight,
                toCryptaIdNeighboursCount,
                fromCryptaIdComponentSize,
                edgeScore,
                scoreGain,
                similarityResults,
                status,
                opposite
        );
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof MergeOffer offer)) {
            return false;
        }

        return Objects.equals(mergeKey, offer.mergeKey) &&
                Objects.equals(toCryptaId, offer.toCryptaId) &&
                Objects.equals(fromCryptaId, offer.fromCryptaId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(mergeKey, toCryptaId, fromCryptaId);
    }

    @Override
    public String toString() {
        return String.format("[%s]%s->%s[%s]",
                mergeKey,
                fromCryptaId,
                toCryptaId,
                status
        );

    }

    @Override
    public int compareTo(MergeOffer o) {
        return BY_MERGE_KEY_COMPARATOR.compare(this, o);
    }
}
