package ru.yandex.crypta.graph2.model.matching.merge;

import java.util.Comparator;

import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.graph2.model.soup.edge.weight.EdgeScore;
import ru.yandex.inside.yt.kosher.impl.operations.utils.YtSerializable;

public class MergeOfferPriority implements YtSerializable {
    // use static fields for comparators to avoid serialization issues
    private static final Comparator<MergeOffer> BEST_COMPARATOR = Comparator
            .comparing(MergeOffer::getEdgeScore, EdgeScore.COMPARATOR)
            .thenComparingDouble(MergeOffer::getScoreGainValue)
            .thenComparing(MergeOffer::getToCryptaId);

    private static final Comparator<MergeOffer> BEST_WITH_FORCE_SINGLE_MERGE_COMPARATOR = Comparator
            .comparing(MergeOfferPriority::isSingleVertexComponentMerge)
            .thenComparing(MergeOffer::getEdgeScore, EdgeScore.COMPARATOR)
            .thenComparingDouble(MergeOffer::getScoreGainValue)
            .thenComparing(MergeOffer::getToCryptaId);

    private static final Comparator<MergeOffer> PRIORITY_COMPARATOR = Comparator
            .comparing(MergeOffer::getEdgeScore, EdgeScore.COMPARATOR)
            .thenComparingDouble(MergeOffer::getScoreGainValue)
            .thenComparing(
                    MergeOffer::getStatus, Comparator.nullsFirst(MergeOfferStatus.BY_STATUS)
            )
            .thenComparing(MergeOffer::getToCryptaId);

    private static final Comparator<MergeOffer> PRIORITY_WITH_FORCE_SINGLE_MERGE_COMPARATOR = Comparator
            .comparing(MergeOfferPriority::isSingleVertexComponentMerge)
            .thenComparing(MergeOffer::getEdgeScore, EdgeScore.COMPARATOR)
            .thenComparingDouble(MergeOffer::getScoreGainValue)
            .thenComparing(
                    MergeOffer::getStatus, Comparator.nullsFirst(MergeOfferStatus.BY_STATUS)
            )
            .thenComparing(MergeOffer::getToCryptaId);

    private final boolean useSingleVertexForceMerge;

    public MergeOfferPriority() {
        this.useSingleVertexForceMerge = false;
    }

    public MergeOfferPriority(boolean useSingleVertexForceMerge) {
        this.useSingleVertexForceMerge = useSingleVertexForceMerge;
    }

    public ListF<MergeOffer> sortMergeOfferByActivityAsc(CollectionF<MergeOffer> offers) {
        if (useSingleVertexForceMerge) {
            return offers.sorted(PRIORITY_WITH_FORCE_SINGLE_MERGE_COMPARATOR);
        } else {
            return offers.sorted(PRIORITY_COMPARATOR);
        }

    }

    public ListF<MergeOffer> sortMergeOfferByActivityDesc(CollectionF<MergeOffer> offers) {
        if (useSingleVertexForceMerge) {
            return offers.sorted(PRIORITY_WITH_FORCE_SINGLE_MERGE_COMPARATOR.reversed());
        } else {
            return offers.sorted(PRIORITY_COMPARATOR.reversed());
        }
    }

    public MergeOffer getBest(CollectionF<MergeOffer> offers) {
        // need stable result not depending on status to coordinate two reducers
        if (useSingleVertexForceMerge) {
            return offers.max(BEST_WITH_FORCE_SINGLE_MERGE_COMPARATOR);
        } else {
            return offers.max(BEST_COMPARATOR);
        }
    }

    public boolean hasBetterLeaderCandidates(ListF<MergeOffer> offers) {

        if (useSingleVertexForceMerge) {
            boolean hasSingleNeighbours = offers.stream().anyMatch(offer ->
                    offer.getToCryptaIdComponentSize() == 1 && offer.getToCryptaIdNeighboursCount() == 1
            );

            boolean singleItself = offers.stream().anyMatch(offer ->
                    offer.getFromCryptaIdComponentSize() == 1 && offer.getFromCryptaIdNeighboursCount() == 1
            );

            if (hasSingleNeighbours && !singleItself) {
                // need to join all these single components first, thus let's treat ourselves as leader
                // when single itself, it's special case of single standalone edge
                return false;
            }
        }

        // TODO: in fact, possible merges count from ComponentMergeProbabilityHeuristic is used to compare components
        return offers.stream().anyMatch(
                offer -> offer.getToCryptaIdComponentWeight() >= offer.getFromCryptaIdComponentWeight()
        );
    }

    private static boolean isSingleVertexComponentMerge(MergeOffer mergeOffer) {
        // hope that single vertex components with no neighbours won't cause transitive merges
        return (mergeOffer.getFromCryptaIdNeighboursCount() == 1 && mergeOffer.getFromCryptaIdComponentSize() == 1)
                || (mergeOffer.getToCryptaIdNeighboursCount() == 1 && mergeOffer.getToCryptaIdComponentSize() == 1);
    }


}
