package ru.yandex.crypta.graph2.model.matching.merge.algo.merge;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.graph.soup.config.proto.TEdgeProps;
import ru.yandex.crypta.graph2.model.matching.component.Component;
import ru.yandex.crypta.graph2.model.matching.component.GraphInfo;
import ru.yandex.crypta.graph2.model.matching.component.score.ComponentScoringStrategy;
import ru.yandex.crypta.graph2.model.matching.component.similarity.ComponentsSimilarityStrategy;
import ru.yandex.crypta.graph2.model.matching.component.similarity.SimilarityResult;
import ru.yandex.crypta.graph2.model.matching.merge.MergeKey;
import ru.yandex.crypta.graph2.model.matching.merge.MergeOffer;
import ru.yandex.crypta.graph2.model.matching.merge.MergeOfferStatus;
import ru.yandex.crypta.graph2.model.matching.score.MetricsTree;
import ru.yandex.crypta.graph2.model.soup.edge.Edge;
import ru.yandex.crypta.graph2.model.soup.edge.weight.EdgeInfoProvider;
import ru.yandex.crypta.graph2.model.soup.edge.weight.EdgeScore;
import ru.yandex.crypta.graph2.model.soup.edge.weight.SurvivalEdgeInfoProvider;
import ru.yandex.inside.yt.kosher.impl.operations.utils.YtSerializable;

public class MergeByScoreAndSimilarityAlgorithm implements MergeAlgorithm {

    private final ComponentScoringStrategy componentScoringStrategy;
    private final ComponentsSimilarityStrategy componentsSimilarityStrategy;
    private final EdgeInfoProvider edgeInfoProvider;
    private final SurvivalEdgeInfoProvider survivalEdgeInfoProvider;
    private final MergeMode mergeMode;

    public static class MergeMode implements YtSerializable {
        public boolean forceIgnoreScores = false;
        public boolean ignoreScoresForGoodLinks = false;
        public int mergeSizeLimit = 2000;
    }

    public MergeByScoreAndSimilarityAlgorithm(ComponentScoringStrategy componentScoringStrategy,
                                              ComponentsSimilarityStrategy componentsSimilarityStrategy,
                                              EdgeInfoProvider edgeInfoProvider) {
        this(componentScoringStrategy, componentsSimilarityStrategy, edgeInfoProvider, new MergeMode());
    }

    public MergeByScoreAndSimilarityAlgorithm(ComponentScoringStrategy componentScoringStrategy,
                                              ComponentsSimilarityStrategy componentsSimilarityStrategy,
                                              EdgeInfoProvider edgeInfoProvider, MergeMode mode) {
        this.componentScoringStrategy = componentScoringStrategy;
        this.componentsSimilarityStrategy = componentsSimilarityStrategy;
        this.edgeInfoProvider = edgeInfoProvider;
        this.survivalEdgeInfoProvider = new SurvivalEdgeInfoProvider();
        this.mergeMode = mode;
    }

    @Override
    public MergeOffer merge(MergeKey mergeKey, Component leftComponent, Component rightComponent,
                            GraphInfo graphInfo, ListF<Edge> betweenEdges) {
        EdgeScore linkScore = edgeInfoProvider.getMultiEdgeScore(betweenEdges);

        MergeOffer mergeOffer = new MergeOffer(
                mergeKey,
                leftComponent.getCryptaId(),
                rightComponent.getCryptaId()
        );

        mergeOffer.setEdgeScore(linkScore);

        mergeOffer.setFromCryptaIdNeighboursCount(leftComponent.getNeighboursCount());
        mergeOffer.setFromCryptaIdComponentWeight(leftComponent.getNeighboursWeight());
        mergeOffer.setFromCryptaIdComponentSize(leftComponent.size());

        mergeOffer.setToCryptaIdNeighboursCount(rightComponent.getNeighboursCount());
        mergeOffer.setToCryptaIdComponentWeight(rightComponent.getNeighboursWeight());
        mergeOffer.setToCryptaIdComponentSize(rightComponent.size());

        if (leftComponent.size() + rightComponent.size() > mergeMode.mergeSizeLimit) {
            // protection from oom and performance degradation
            mergeOffer.setScoreGain(new MetricsTree(-100, "oversize"));

            return mergeOffer.withStatus(MergeOfferStatus.FAILED_BY_SCORE);

        } else if (linkScore.getStrength().equals(TEdgeProps.EEdgeStrength.ARTIFICIAL)
                || linkScore.getStrength().equals(TEdgeProps.EEdgeStrength.TRUSTED)) {
            // trusted edge connects component no matter of score
            return mergeOffer.withStatus(MergeOfferStatus.INIT_TRUSTED);

        } else {
            // all failed edges connect component only if the make good score
            leftComponent.computeTGraph(graphInfo);
            rightComponent.computeTGraph(graphInfo);
            Component merged = leftComponent.mergeWithGraphEngine(rightComponent, betweenEdges);

            componentScoringStrategy.setScoreTree(merged, graphInfo);
            // we can pre-compute it
            componentScoringStrategy.setScoreTreeWeakForce(leftComponent, graphInfo);
            componentScoringStrategy.setScoreTreeWeakForce(rightComponent, graphInfo);

            MetricsTree scoreGain = MetricsTree.calculateScoreGain(
                    merged.getScoreTree(),
                    Cf.list(leftComponent.getScoreTree(), rightComponent.getScoreTree())
            );

            ListF<SimilarityResult> similarityResults = componentsSimilarityStrategy.isSimilar(
                    leftComponent, rightComponent, graphInfo
            );

            mergeOffer.setScoreGain(scoreGain);
            mergeOffer.setSimilarityResults(similarityResults);

            if (!ignoreScore(betweenEdges, linkScore) && scoreGain.getScore() < Double.MIN_VALUE) { // to fix double precision
                return mergeOffer.withStatus(MergeOfferStatus.FAILED_BY_SCORE);
            }

            boolean canIgnoreSimilarity = ignoreSimilarityForGoodLinks(betweenEdges, linkScore);
            boolean notSimilar = similarityResults.filter(
                    sr -> !sr.isSimilar() && !(canIgnoreSimilarity && sr.isWeak())
            ).isNotEmpty();
            if (notSimilar) {
                return mergeOffer.withStatus(MergeOfferStatus.FAILED_BY_SIMILARITY);
            }

            return mergeOffer.withStatus(MergeOfferStatus.INIT);

        }
    }

    boolean ignoreScore(ListF<Edge> betweenEdges, EdgeScore linkScore) {
        if (mergeMode.forceIgnoreScores) {
            return true;
        }
        if (!mergeMode.ignoreScoresForGoodLinks) {
            return false;
        }
        double survivalCutWeight = survivalEdgeInfoProvider.getMultiEdgeWeight(betweenEdges);
        return survivalEdgeInfoProvider.isStrong(linkScore.getStrength()) || betweenEdges.size() > 2 ||
                (betweenEdges.size() > 1 && (survivalCutWeight >= 1 || linkScore.getWeight() > 1));
        //  Гипотеза такая : если есть хотя бы одно ребро с сильной связью или 3 ребра с какой-то,
        //  то стоит попробовать компоненты смерджить вне зависимости от основного GainScore
        //  (два ребра мало, они могут еще быть случайными про одно и то же, довольно частый кейс).
        //  GainScore мог оказаться при этом плохим из-за неправильной склейки до этого, и тогда
        //  мы надеемся, что в Split стадии полученная компонента будет разделена более релевантно.
    }

    boolean ignoreSimilarityForGoodLinks(ListF<Edge> betweenEdges, EdgeScore linkScore) {
        double survivalCutWeight = survivalEdgeInfoProvider.getMultiEdgeWeight(betweenEdges);
        return (survivalEdgeInfoProvider.isStrong(linkScore.getStrength()) &&  betweenEdges.size() > 3)
                || betweenEdges.size() > 5
                || survivalCutWeight >= 3;
    }
}
