package ru.yandex.crypta.graph2.model.matching.merge.algo.score;

import java.util.Set;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.crypta.graph2.model.matching.component.Component;
import ru.yandex.crypta.graph2.model.matching.component.GraphInfo;
import ru.yandex.crypta.graph2.model.matching.score.MetricsTree;
import ru.yandex.crypta.graph2.model.soup.edge.Edge;
import ru.yandex.crypta.graph2.model.soup.edge.weight.EdgeInfoProvider;
import ru.yandex.crypta.graph2.model.soup.vertex.Vertex;


public class WeightedLinkScoringStrategy implements LinkScoringStrategy {

    public static final String SPLIT_SCORE_CODE = "split";
    public static final double SPLIT_SCORE_FOR_FORCE_SPLITTING = 10;
    public static final double SPLIT_SCORE_FOR_SMALL_COMPONENT = -1e-4;

    private static final double MIN_EDGE_PROBABILITY = 1e-2;
    private static final double THRESHOLD_FOR_STRONG_WEIGHT = .5;

    private final EdgeInfoProvider edgeInfoProvider;
    private final int maxVerticesCount;
    private final int maxEdgesCount;
    private final int minReliableComponentSize;
    private final int minPositiveComponentSize;

    public WeightedLinkScoringStrategy(EdgeInfoProvider edgeInfoProvider, int maxVerticesCount, int maxEdgesCount,
                                       int minReliableComponentSize, int minPositiveComponentSize) {
        this.edgeInfoProvider = edgeInfoProvider;
        this.maxVerticesCount = maxVerticesCount;
        this.maxEdgesCount = maxEdgesCount;
        this.minReliableComponentSize = minReliableComponentSize;
        this.minPositiveComponentSize = minPositiveComponentSize;
    }

    private boolean isComponentBig(int verticesCount, int edgesCount) {
        return verticesCount > maxVerticesCount || edgesCount > maxEdgesCount;
    }

    @Override
    public boolean isCutInteresting(ListF<Set<Vertex>> connectedSets) {
        if (connectedSets.size() < 2) {
            return false;
        }
        Set<Vertex> secondLargest = connectedSets.sortedByDesc(Set::size).get(1);
        return secondLargest.size() >= minPositiveComponentSize;
    }

    @Override
    public MetricsTree getScore(CollectionF<Component> components, ListF<Edge> betweenEdges, GraphInfo graphInfo) {
        if (isComponentBig(components.map(Component::size).sum(Cf.Integer),
                components.map(c -> c.getInnerEdges().size()).sum(Cf.Integer) + betweenEdges.size())) {
            return new MetricsTree(SPLIT_SCORE_FOR_FORCE_SPLITTING, SPLIT_SCORE_CODE);
        }
        double score = 1.;
        for (Edge edge : betweenEdges) {
            double weight = edgeInfoProvider.getEdgeWeight(edge);
            if (edgeInfoProvider.isStrong(edge) || weight > THRESHOLD_FOR_STRONG_WEIGHT) {
                score = -1;
                break;
            }
            score *= (1. - Math.min(1., Math.max(weight, MIN_EDGE_PROBABILITY)));
        }
        if (components.map(Component::size).min() >= minReliableComponentSize) {
            // try split by link score
            return new MetricsTree(score, SPLIT_SCORE_CODE);
        } else if (components.map(Component::size).min() < minPositiveComponentSize) {
            // ask caller not to split
            return new MetricsTree(Math.min(score, SPLIT_SCORE_FOR_SMALL_COMPONENT), SPLIT_SCORE_CODE);
        } else {
            // no decision, don't interfere
            return new MetricsTree(Math.min(score, 0), SPLIT_SCORE_CODE);
        }

    }
}

