package ru.yandex.crypta.graph2.model.matching.merge.algo.split;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.crypta.graph2.model.matching.component.Component;
import ru.yandex.crypta.graph2.model.matching.component.GraphInfo;
import ru.yandex.crypta.graph2.model.matching.component.score.ComponentScoringStrategy;
import ru.yandex.crypta.graph2.model.matching.merge.MergeOfferStatus;
import ru.yandex.crypta.graph2.model.matching.merge.algo.score.LinkScoringStrategy;
import ru.yandex.crypta.graph2.model.matching.merge.algo.split.inner.ComponentConnectivityInspector;
import ru.yandex.crypta.graph2.model.matching.merge.algo.split.inner.ComponentSplitDendrogram;
import ru.yandex.crypta.graph2.model.matching.score.MetricsTree;
import ru.yandex.crypta.graph2.model.soup.edge.Edge;

public abstract class AbstractSplitAlgorithm implements SplitAlgorithm {

    private static final int MAX_SPLIT_DEPTH = 100;

    protected final ComponentScoringStrategy componentScoringStrategy;
    protected final LinkScoringStrategy linkScoringStrategy;

    protected AbstractSplitAlgorithm(ComponentScoringStrategy componentScoringStrategy,
            LinkScoringStrategy linkScoringStrategy)
    {
        this.componentScoringStrategy = componentScoringStrategy;
        this.linkScoringStrategy = linkScoringStrategy;
    }

    protected abstract ListF<Edge> prepareRemovalCandidates(Component component);

    protected abstract ListF<Edge> chooseRemovalEdges(Component component, ListF<Edge> removalCandidates);


    private Tuple2<MergeOfferStatus, MetricsTree> makeSplitDecision(Component component,
            ListF<Component> splitComponents, ListF<Edge> edgesBetween, GraphInfo graphInfo)
    {
        componentScoringStrategy.setScoreTree(component, graphInfo);
        componentScoringStrategy.setScoreTree(splitComponents, graphInfo);


        MetricsTree splitScoreGain = linkScoringStrategy.getScore(splitComponents, edgesBetween, graphInfo);
        MetricsTree componentsScoreGain = MetricsTree.calculateScoreGain(
                component.getScoreTree(),
                splitComponents.map(Component::getScoreTree)
        );

        MetricsTree scoreGain = componentsScoreGain.minus(splitScoreGain);

        if (scoreGain.getScore() < Double.MIN_VALUE) {
            return Tuple2.tuple(MergeOfferStatus.SPLIT, scoreGain);
        } else {
            return Tuple2.tuple(MergeOfferStatus.FAILED_BY_SCORE, scoreGain);
        }
    }


    public Option<ComponentSplitDendrogram> getSplit(Component component, ListF<Edge> removalCandidates,
            GraphInfo graphInfo)
    {
        ComponentConnectivityInspector connectivityInspector = new ComponentConnectivityInspector(component);
        ListF<Edge> removalEdges =
                chooseRemovalEdges(component, removalCandidates.filter(component.getInnerEdges()::containsTs));
        if (removalEdges.size() == 0) {
            return Option.empty();
        }
        ListF<Component> components = connectivityInspector.getSplit(removalEdges);
        removalEdges = connectivityInspector.getRemovedEdges();

        if (components.size() != 2) {
            return Option.empty();
        }

        Component leftComponent = components.get(0);
        Component rightComponent = components.get(1);

        if (component.size() == leftComponent.size() || component.size() == rightComponent.size()) {
            return Option.empty();
        }

        Tuple2<MergeOfferStatus, MetricsTree> splitScore =
                makeSplitDecision(component, components, removalEdges, graphInfo);

        ComponentSplitDendrogram split = new ComponentSplitDendrogram(
                leftComponent,
                rightComponent,
                splitScore._2,
                removalEdges
        ).withStatus(splitScore._1);

        return Option.of(split);
    }

    private Option<ComponentSplitDendrogram> tryRecursivelySplitComponent(Component component, GraphInfo graphInfo,
            ListF<Edge> removalCandidates, int level) {

        if (level >= MAX_SPLIT_DEPTH) {
            return Option.empty();
        }

        Option<ComponentSplitDendrogram> split = getSplit(component, removalCandidates, graphInfo);
        if (split.isPresent()) {
            if (split.get().getStatus() == MergeOfferStatus.SPLIT) {
                Option<ComponentSplitDendrogram> leftSplit = tryRecursivelySplitComponent(
                        split.get().getLeftComponent(),
                        graphInfo,
                        removalCandidates,
                        level + 1
                );
                Option<ComponentSplitDendrogram> rightSplit = tryRecursivelySplitComponent(
                        split.get().getRightComponent(),
                        graphInfo,
                        removalCandidates,
                        level + 1
                );
                return Option.of(split.get().withLeftSplit(leftSplit).withRightSplit(rightSplit));
            } else {
                return split;
            }
        }
        return Option.empty();
    }


    @Override
    public Option<ComponentSplitDendrogram> split(Component component, GraphInfo graphInfo) {
        ListF<Edge> removalCandidates = prepareRemovalCandidates(component);
        return tryRecursivelySplitComponent(component, graphInfo, removalCandidates, 0);

    }
}
